import { buildUrl, toServiceException, toPayload } from './_util';
import { StructuredError } from '../error';
const KeyValueInvalidTTLError = StructuredError('KeyValueInvalidTTLError');
export class KeyValueStorageService {
    #adapter;
    #baseUrl;
    constructor(baseUrl, adapter) {
        this.#adapter = adapter;
        this.#baseUrl = baseUrl;
    }
    async get(name, key) {
        const url = buildUrl(this.#baseUrl, `/kv/2025-03-17/${encodeURIComponent(name)}/${encodeURIComponent(key)}`);
        const signal = AbortSignal.timeout(10_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.keyvalue.get',
                attributes: {
                    name,
                    key,
                },
            },
        });
        if (res.ok) {
            return {
                data: res.data,
                contentType: res.response.headers.get('content-type') ?? 'application/octet-stream',
                exists: true,
            };
        }
        if (res.response.status === 404) {
            return { exists: false };
        }
        throw await toServiceException('GET', url, res.response);
    }
    async set(name, key, value, params) {
        let ttlstr = '';
        if (params?.ttl) {
            if (params.ttl < 60) {
                throw new KeyValueInvalidTTLError({
                    message: `ttl for keyvalue set must be at least 60 seconds, got ${params.ttl}`,
                });
            }
            ttlstr = `/${params.ttl}`;
        }
        const url = buildUrl(this.#baseUrl, `/kv/2025-03-17/${encodeURIComponent(name)}/${encodeURIComponent(key)}${ttlstr}`);
        const [body, contentType] = await toPayload(value);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'PUT',
            signal,
            body,
            contentType: params?.contentType || contentType,
            telemetry: {
                name: 'agentuity.keyvalue.set',
                attributes: {
                    name,
                    key,
                    ttl: ttlstr,
                },
            },
        });
        if (res.ok) {
            return;
        }
        throw await toServiceException('PUT', url, res.response);
    }
    async delete(name, key) {
        const url = buildUrl(this.#baseUrl, `/kv/2025-03-17/${encodeURIComponent(name)}/${encodeURIComponent(key)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'DELETE',
            signal,
            telemetry: {
                name: 'agentuity.keyvalue.delete',
                attributes: {
                    name,
                    key,
                },
            },
        });
        if (res.ok) {
            return;
        }
        throw await toServiceException('DELETE', url, res.response);
    }
    async getStats(name) {
        const url = buildUrl(this.#baseUrl, `/kv/2025-03-17/stats/${encodeURIComponent(name)}`);
        const signal = AbortSignal.timeout(10_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.keyvalue.getStats',
                attributes: { name },
            },
        });
        if (res.ok) {
            return res.data;
        }
        throw await toServiceException('GET', url, res.response);
    }
    async getAllStats() {
        const url = buildUrl(this.#baseUrl, '/kv/2025-03-17/stats');
        const signal = AbortSignal.timeout(10_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.keyvalue.getAllStats',
                attributes: {},
            },
        });
        if (res.ok) {
            return res.data;
        }
        throw await toServiceException('GET', url, res.response);
    }
    async getNamespaces() {
        const stats = await this.getAllStats();
        return Object.keys(stats);
    }
    async search(name, keyword) {
        const url = buildUrl(this.#baseUrl, `/kv/2025-03-17/search/${encodeURIComponent(name)}/${encodeURIComponent(keyword)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.keyvalue.search',
                attributes: { name, keyword },
            },
        });
        if (res.ok) {
            return res.data;
        }
        throw await toServiceException('GET', url, res.response);
    }
    async getKeys(name) {
        const url = buildUrl(this.#baseUrl, `/kv/2025-03-17/keys/${encodeURIComponent(name)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.keyvalue.getKeys',
                attributes: { name },
            },
        });
        if (res.ok) {
            return res.data;
        }
        throw await toServiceException('GET', url, res.response);
    }
    async deleteNamespace(name) {
        const url = buildUrl(this.#baseUrl, `/kv/2025-03-17/${encodeURIComponent(name)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'DELETE',
            signal,
            telemetry: {
                name: 'agentuity.keyvalue.deleteNamespace',
                attributes: { name },
            },
        });
        if (res.ok) {
            return;
        }
        throw await toServiceException('DELETE', url, res.response);
    }
    async createNamespace(name) {
        const url = buildUrl(this.#baseUrl, `/kv/2025-03-17/${encodeURIComponent(name)}`);
        const signal = AbortSignal.timeout(10_000);
        const res = await this.#adapter.invoke(url, {
            method: 'POST',
            signal,
            telemetry: {
                name: 'agentuity.keyvalue.createNamespace',
                attributes: { name },
            },
        });
        if (res.ok) {
            return;
        }
        throw await toServiceException('POST', url, res.response);
    }
}
//# sourceMappingURL=keyvalue.js.map