import { FetchAdapter } from './adapter';
/**
 * the result of a data operation when the data is found
 */
export interface DataResultFound<T> {
    /**
     * the data from the result of the operation
     */
    data: T;
    /**
     * the content type of the data
     */
    contentType: string;
    /**
     * the data was found
     */
    exists: true;
}
/**
 * the result of a data operation when the data is not found
 */
export interface DataResultNotFound {
    data: never;
    /**
     * the data was not found
     */
    exists: false;
}
/**
 * the result of a data operation
 */
export type DataResult<T> = DataResultFound<T> | DataResultNotFound;
export interface KeyValueStorageSetParams {
    /**
     * the number of milliseconds to keep the value in the cache
     */
    ttl?: number;
    /**
     * the content type of the value
     */
    contentType?: string;
}
/**
 * Statistics for a key-value store namespace
 */
export interface KeyValueStats {
    sum: number;
    count: number;
    createdAt?: number;
    lastUsedAt?: number;
}
/**
 * Item with metadata from search results
 */
export interface KeyValueItemWithMetadata<T = unknown> {
    value: T;
    contentType: string;
    size: number;
    created_at: string;
    updated_at: string;
}
export interface KeyValueStorage {
    /**
     * get a value from the key value storage
     *
     * @param name - the name of the key value storage
     * @param key - the key to get the value of
     * @returns the DataResult object
     */
    get<T>(name: string, key: string): Promise<DataResult<T>>;
    /**
     * set a value in the key value storage
     *
     * @param name - the name of the key value storage
     * @param key - the key to set the value of
     * @param value - the value to set in any of the supported data types
     * @param params - the KeyValueStorageSetParams
     */
    set<T = unknown>(name: string, key: string, value: T, params?: KeyValueStorageSetParams): Promise<void>;
    /**
     * delete a value from the key value storage
     *
     * @param name - the name of the key value storage
     * @param key - the key to delete
     */
    delete(name: string, key: string): Promise<void>;
    /**
     * get statistics for a specific namespace
     *
     * @param name - the name of the key value storage
     * @returns statistics for the namespace
     */
    getStats(name: string): Promise<KeyValueStats>;
    /**
     * get statistics for all namespaces
     *
     * @returns map of namespace names to statistics
     */
    getAllStats(): Promise<Record<string, KeyValueStats>>;
    /**
     * get all namespace names
     *
     * @returns array of namespace names
     */
    getNamespaces(): Promise<string[]>;
    /**
     * search for keys matching a keyword
     *
     * @param name - the name of the key value storage
     * @param keyword - the keyword to search for
     * @returns map of keys to items with metadata
     */
    search<T = unknown>(name: string, keyword: string): Promise<Record<string, KeyValueItemWithMetadata<T>>>;
    /**
     * get all keys in a namespace
     *
     * @param name - the name of the key value storage
     * @returns array of keys
     */
    getKeys(name: string): Promise<string[]>;
    /**
     * delete all keys in a namespace
     *
     * @param name - the name of the key value storage
     */
    deleteNamespace(name: string): Promise<void>;
    /**
     * create a new namespace
     *
     * @param name - the name of the key value storage to create
     */
    createNamespace(name: string): Promise<void>;
}
export declare class KeyValueStorageService implements KeyValueStorage {
    #private;
    constructor(baseUrl: string, adapter: FetchAdapter);
    get<T>(name: string, key: string): Promise<DataResult<T>>;
    set<T = unknown>(name: string, key: string, value: T, params?: KeyValueStorageSetParams): Promise<void>;
    delete(name: string, key: string): Promise<void>;
    getStats(name: string): Promise<KeyValueStats>;
    getAllStats(): Promise<Record<string, KeyValueStats>>;
    getNamespaces(): Promise<string[]>;
    search<T = unknown>(name: string, keyword: string): Promise<Record<string, KeyValueItemWithMetadata<T>>>;
    getKeys(name: string): Promise<string[]>;
    deleteNamespace(name: string): Promise<void>;
    createNamespace(name: string): Promise<void>;
}
//# sourceMappingURL=keyvalue.d.ts.map