import { parseArgsSchema, parseOptionsSchema } from './schema-parser';
import * as z from 'zod';
/**
 * Extract schema information from a CommandDefinition
 */
function extractCommandSchema(def) {
    const schema = {
        name: def.name,
        description: def.description,
    };
    if (def.aliases) {
        schema.aliases = def.aliases;
    }
    // Extract examples if available
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    if (def.examples) {
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        schema.examples = def.examples;
    }
    // Extract idempotent marker
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    if (def.idempotent !== undefined) {
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        schema.idempotent = def.idempotent;
    }
    // Extract prerequisites
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    if (def.prerequisites) {
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        schema.prerequisites = def.prerequisites;
    }
    // Extract pagination
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    if (def.pagination) {
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        schema.pagination = def.pagination;
    }
    // Extract tags
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    if (def.tags) {
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        schema.tags = def.tags;
    }
    // Extract skipSkill
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    if (def.skipSkill !== undefined) {
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        schema.skipSkill = def.skipSkill;
    }
    // Extract requires/optional
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    const d = def;
    if (d.requires) {
        schema.requires = {
            auth: d.requires.auth === true,
            project: d.requires.project === true,
            org: d.requires.org === true,
            region: d.requires.region === true,
            regions: d.requires.regions === true,
        };
    }
    if (d.optional) {
        schema.optional = {
            auth: d.optional.auth === true || typeof d.optional.auth === 'string',
            project: d.optional.project === true,
            org: d.optional.org === true,
            region: d.optional.region === true,
        };
    }
    // Extract subcommands recursively
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    if (def.subcommands) {
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        schema.subcommands = def.subcommands.map((sub) => extractSubcommandSchema(sub));
    }
    return schema;
}
/**
 * Extract schema information from a SubcommandDefinition
 */
function extractSubcommandSchema(def) {
    const schema = {
        name: def.name,
        description: def.description,
    };
    if (def.aliases) {
        schema.aliases = def.aliases;
    }
    // Extract examples if available
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    if (def.examples) {
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        schema.examples = def.examples;
    }
    // Extract idempotent marker
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    if (def.idempotent !== undefined) {
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        schema.idempotent = def.idempotent;
    }
    // Extract prerequisites
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    if (def.prerequisites) {
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        schema.prerequisites = def.prerequisites;
    }
    // Extract pagination
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    if (def.pagination) {
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        schema.pagination = def.pagination;
    }
    // Extract tags
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    if (def.tags) {
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        schema.tags = def.tags;
    }
    // Extract skipSkill
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    if (def.skipSkill !== undefined) {
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        schema.skipSkill = def.skipSkill;
    }
    // Extract toplevel
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    if (def.toplevel !== undefined) {
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        schema.toplevel = def.toplevel;
    }
    // Extract requires/optional
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    const d = def;
    if (d.requires) {
        schema.requires = {
            auth: d.requires.auth === true,
            project: d.requires.project === true,
            org: d.requires.org === true,
            region: d.requires.region === true,
            regions: d.requires.regions === true,
        };
    }
    if (d.optional) {
        schema.optional = {
            auth: d.optional.auth === true || typeof d.optional.auth === 'string',
            project: d.optional.project === true,
            org: d.optional.org === true,
            region: d.optional.region === true,
        };
    }
    // Extract args and options from schema if available
    if (d.schema) {
        const schemas = d.schema;
        if (schemas.args) {
            const parsedArgs = parseArgsSchema(schemas.args);
            schema.arguments = parsedArgs.metadata.map((arg) => ({
                name: arg.name,
                type: arg.variadic ? 'array' : 'string',
                required: !arg.optional,
                variadic: arg.variadic,
            }));
        }
        if (schemas.options) {
            const parsedOptions = parseOptionsSchema(schemas.options);
            schema.options = parsedOptions.map((opt) => ({
                name: opt.name,
                type: opt.type,
                required: !opt.hasDefault,
                default: opt.defaultValue,
                description: opt.description,
            }));
        }
        if (schemas.response) {
            schema.response = z.toJSONSchema(schemas.response);
        }
    }
    // Extract nested subcommands recursively
    if (d.subcommands) {
        schema.subcommands = d.subcommands.map((sub) => extractSubcommandSchema(sub));
    }
    return schema;
}
/**
 * Generate JSON schema for the entire CLI
 */
export function generateCLISchema(program, commands, version) {
    const schema = {
        name: 'agentuity',
        version,
        description: 'Agentuity CLI',
        exitCodes: {
            0: 'Success',
            1: 'General error',
            2: 'Validation error (invalid arguments or options)',
            3: 'Authentication error (login required or credentials invalid)',
            4: 'Resource not found (project, file, deployment, etc.)',
            5: 'Permission denied (insufficient access rights)',
            6: 'Network error (API unreachable or timeout)',
            7: 'File system error (file read/write failed)',
            8: 'User cancelled (operation aborted by user)',
            9: 'Build failed',
        },
        globalOptions: [
            {
                name: 'config',
                type: 'string',
                required: false,
                description: 'Config file path',
            },
            {
                name: 'log-level',
                type: 'string',
                required: false,
                default: 'info',
                description: 'Log level',
                enum: ['debug', 'trace', 'info', 'warn', 'error'],
            },
            {
                name: 'log-timestamp',
                type: 'boolean',
                required: false,
                default: false,
                description: 'Show timestamps in log output',
            },
            {
                name: 'log-prefix',
                type: 'boolean',
                required: false,
                default: true,
                description: 'Show log level prefixes',
            },
            {
                name: 'org-id',
                type: 'string',
                required: false,
                description: 'Use a specific organization when performing operations',
            },
            {
                name: 'color-scheme',
                type: 'string',
                required: false,
                description: 'Color scheme: light or dark',
                enum: ['light', 'dark'],
            },
            {
                name: 'color',
                type: 'string',
                required: false,
                default: 'auto',
                description: 'Color output: auto, always, never',
                enum: ['auto', 'always', 'never'],
            },
            {
                name: 'error-format',
                type: 'string',
                required: false,
                default: 'text',
                description: 'Error output format: json or text',
                enum: ['json', 'text'],
            },
            {
                name: 'json',
                type: 'boolean',
                required: false,
                default: false,
                description: 'Output in JSON format (machine-readable)',
            },
            {
                name: 'quiet',
                type: 'boolean',
                required: false,
                default: false,
                description: 'Suppress non-essential output',
            },
            {
                name: 'no-progress',
                type: 'boolean',
                required: false,
                default: false,
                description: 'Disable progress indicators',
            },
            {
                name: 'explain',
                type: 'boolean',
                required: false,
                default: false,
                description: 'Show what the command would do without executing',
            },
            {
                name: 'dry-run',
                type: 'boolean',
                required: false,
                default: false,
                description: 'Execute command without making changes',
            },
            {
                name: 'validate',
                type: 'boolean',
                required: false,
                default: false,
                description: 'Validate arguments and options without executing',
            },
        ],
        commands: commands.map(extractCommandSchema),
    };
    return schema;
}
//# sourceMappingURL=schema-generator.js.map