/**
 * Standard exit codes for the CLI
 */
export var ExitCode;
(function (ExitCode) {
    ExitCode[ExitCode["SUCCESS"] = 0] = "SUCCESS";
    ExitCode[ExitCode["GENERAL_ERROR"] = 1] = "GENERAL_ERROR";
    ExitCode[ExitCode["VALIDATION_ERROR"] = 2] = "VALIDATION_ERROR";
    ExitCode[ExitCode["AUTH_ERROR"] = 3] = "AUTH_ERROR";
    ExitCode[ExitCode["NOT_FOUND"] = 4] = "NOT_FOUND";
    ExitCode[ExitCode["PERMISSION_ERROR"] = 5] = "PERMISSION_ERROR";
    ExitCode[ExitCode["NETWORK_ERROR"] = 6] = "NETWORK_ERROR";
    ExitCode[ExitCode["FILE_ERROR"] = 7] = "FILE_ERROR";
    ExitCode[ExitCode["USER_CANCELLED"] = 8] = "USER_CANCELLED";
    ExitCode[ExitCode["BUILD_FAILED"] = 9] = "BUILD_FAILED";
})(ExitCode || (ExitCode = {}));
/**
 * Standard error codes for the CLI
 */
export var ErrorCode;
(function (ErrorCode) {
    // Validation errors
    ErrorCode["VALIDATION_FAILED"] = "VALIDATION_FAILED";
    ErrorCode["MISSING_ARGUMENT"] = "MISSING_ARGUMENT";
    ErrorCode["INVALID_ARGUMENT"] = "INVALID_ARGUMENT";
    ErrorCode["INVALID_OPTION"] = "INVALID_OPTION";
    ErrorCode["UNKNOWN_COMMAND"] = "UNKNOWN_COMMAND";
    // Authentication errors
    ErrorCode["AUTH_REQUIRED"] = "AUTH_REQUIRED";
    ErrorCode["AUTH_FAILED"] = "AUTH_FAILED";
    ErrorCode["AUTH_EXPIRED"] = "AUTH_EXPIRED";
    ErrorCode["PERMISSION_DENIED"] = "PERMISSION_DENIED";
    // Project/Configuration errors
    ErrorCode["PROJECT_NOT_FOUND"] = "PROJECT_NOT_FOUND";
    ErrorCode["PROJECT_INVALID"] = "PROJECT_INVALID";
    ErrorCode["CONFIG_NOT_FOUND"] = "CONFIG_NOT_FOUND";
    ErrorCode["CONFIG_INVALID"] = "CONFIG_INVALID";
    // Resource errors
    ErrorCode["RESOURCE_NOT_FOUND"] = "RESOURCE_NOT_FOUND";
    ErrorCode["RESOURCE_ALREADY_EXISTS"] = "RESOURCE_ALREADY_EXISTS";
    ErrorCode["RESOURCE_CONFLICT"] = "RESOURCE_CONFLICT";
    // Organization/Region errors
    ErrorCode["ORG_REQUIRED"] = "ORG_REQUIRED";
    ErrorCode["ORG_NOT_FOUND"] = "ORG_NOT_FOUND";
    ErrorCode["REGION_REQUIRED"] = "REGION_REQUIRED";
    ErrorCode["REGION_NOT_FOUND"] = "REGION_NOT_FOUND";
    ErrorCode["NO_REGIONS_AVAILABLE"] = "NO_REGIONS_AVAILABLE";
    // Network/API errors
    ErrorCode["NETWORK_ERROR"] = "NETWORK_ERROR";
    ErrorCode["API_ERROR"] = "API_ERROR";
    ErrorCode["TIMEOUT"] = "TIMEOUT";
    // File system errors
    ErrorCode["FILE_NOT_FOUND"] = "FILE_NOT_FOUND";
    ErrorCode["FILE_READ_ERROR"] = "FILE_READ_ERROR";
    ErrorCode["FILE_WRITE_ERROR"] = "FILE_WRITE_ERROR";
    ErrorCode["DIRECTORY_NOT_FOUND"] = "DIRECTORY_NOT_FOUND";
    // Runtime errors
    ErrorCode["RUNTIME_ERROR"] = "RUNTIME_ERROR";
    ErrorCode["INTERNAL_ERROR"] = "INTERNAL_ERROR";
    ErrorCode["NOT_IMPLEMENTED"] = "NOT_IMPLEMENTED";
    // User cancellation
    ErrorCode["USER_CANCELLED"] = "USER_CANCELLED";
    // Build failed error
    ErrorCode["BUILD_FAILED"] = "BUILD_FAILED";
    // Integration errors
    ErrorCode["INTEGRATION_FAILED"] = "INTEGRATION_FAILED";
})(ErrorCode || (ErrorCode = {}));
/**
 * Map error codes to exit codes
 */
export function getExitCode(errorCode) {
    switch (errorCode) {
        // Validation errors
        case ErrorCode.VALIDATION_FAILED:
        case ErrorCode.MISSING_ARGUMENT:
        case ErrorCode.INVALID_ARGUMENT:
        case ErrorCode.INVALID_OPTION:
        case ErrorCode.UNKNOWN_COMMAND:
        case ErrorCode.PROJECT_INVALID:
        case ErrorCode.CONFIG_INVALID:
            return ExitCode.VALIDATION_ERROR;
        // Authentication errors
        case ErrorCode.AUTH_REQUIRED:
        case ErrorCode.AUTH_FAILED:
        case ErrorCode.AUTH_EXPIRED:
            return ExitCode.AUTH_ERROR;
        // Permission errors
        case ErrorCode.PERMISSION_DENIED:
            return ExitCode.PERMISSION_ERROR;
        // Not found errors
        case ErrorCode.PROJECT_NOT_FOUND:
        case ErrorCode.CONFIG_NOT_FOUND:
        case ErrorCode.RESOURCE_NOT_FOUND:
        case ErrorCode.ORG_NOT_FOUND:
        case ErrorCode.REGION_NOT_FOUND:
        case ErrorCode.FILE_NOT_FOUND:
        case ErrorCode.DIRECTORY_NOT_FOUND:
        case ErrorCode.NO_REGIONS_AVAILABLE:
            return ExitCode.NOT_FOUND;
        // Network/API errors
        case ErrorCode.NETWORK_ERROR:
        case ErrorCode.API_ERROR:
        case ErrorCode.TIMEOUT:
            return ExitCode.NETWORK_ERROR;
        // File system errors
        case ErrorCode.FILE_READ_ERROR:
        case ErrorCode.FILE_WRITE_ERROR:
            return ExitCode.FILE_ERROR;
        // User cancellation
        case ErrorCode.USER_CANCELLED:
            return ExitCode.USER_CANCELLED;
        // Build errors
        case ErrorCode.BUILD_FAILED:
            return ExitCode.BUILD_FAILED;
        // Integration errors
        case ErrorCode.INTEGRATION_FAILED:
            return ExitCode.NETWORK_ERROR;
        // Resource conflicts and other errors
        case ErrorCode.RESOURCE_ALREADY_EXISTS:
        case ErrorCode.RESOURCE_CONFLICT:
        case ErrorCode.ORG_REQUIRED:
        case ErrorCode.REGION_REQUIRED:
        case ErrorCode.RUNTIME_ERROR:
        case ErrorCode.INTERNAL_ERROR:
        case ErrorCode.NOT_IMPLEMENTED:
        default:
            return ExitCode.GENERAL_ERROR;
    }
}
/**
 * Format error in JSON structure
 */
export function formatErrorJSON(error) {
    const exitCode = error.exitCode ?? getExitCode(error.code);
    const output = {
        error: {
            code: error.code,
            message: error.message,
            exitCode,
        },
    };
    if (error.details) {
        output.error.details = error.details;
    }
    if (error.suggestions && error.suggestions.length > 0) {
        output.error.suggestions = error.suggestions;
    }
    return JSON.stringify(output, null, 2);
}
/**
 * Format error for human-readable output
 */
export function formatErrorHuman(error) {
    let output = `error: ${error.message}`;
    if (error.details && Object.keys(error.details).length > 0) {
        output += '\n\nDetails:';
        for (const [key, value] of Object.entries(error.details)) {
            output += `\n  ${key}: ${JSON.stringify(value)}`;
        }
    }
    if (error.suggestions && error.suggestions.length > 0) {
        output += '\n\nSuggestions:';
        for (const suggestion of error.suggestions) {
            output += `\n  - ${suggestion}`;
        }
    }
    return output;
}
/**
 * Exit the process with a structured error
 */
export function exitWithError(error, logger, errorFormat = 'text') {
    if (errorFormat === 'json') {
        console.error(formatErrorJSON(error));
    }
    else {
        logger.error(formatErrorHuman(error));
    }
    const exitCode = error.exitCode ?? getExitCode(error.code);
    process.exit(exitCode);
}
/**
 * Create a structured error from a simple message and code
 */
export function createError(code, message, details, suggestions) {
    return { code, message, details, suggestions };
}
//# sourceMappingURL=errors.js.map