/**
 * Utility functions for handling .env files
 */
export interface EnvVars {
    [key: string]: string;
}
/**
 * Find an existing .env file for reading.
 */
export declare function findExistingEnvFile(dir: string): Promise<string>;
/**
 * Find env files to load based on config profile.
 * Returns list of .env file paths in order of precedence:
 * - For production or non-local: .env.{configName}, .env, .env.production
 * - For local/development: .env.{configName}, .env.development, .env
 */
export declare function getEnvFilePaths(dir: string, options: {
    configName?: string;
    isProduction?: boolean;
}): string[];
/**
 * Parse a single line from an .env file
 * Handles comments, empty lines, and quoted values
 */
export declare function parseEnvLine(line: string): {
    key: string;
    value: string;
} | null;
/**
 * Read and parse an .env file
 */
export declare function readEnvFile(path: string): Promise<EnvVars>;
/**
 * Write environment variables to an .env file
 * Optionally skip certain keys (like AGENTUITY_SDK_KEY)
 */
export declare function writeEnvFile(path: string, vars: EnvVars, options?: {
    skipKeys?: string[];
    addComment?: (key: string) => string | null;
}): Promise<void>;
/**
 * Merge environment variables with special handling
 * - Later values override earlier values
 * - Can filter out keys (like AGENTUITY_* keys)
 */
export declare function mergeEnvVars(base: EnvVars, updates: EnvVars, options?: {
    filterPrefix?: string;
}): EnvVars;
/**
 * Filter out AGENTUITY_ prefixed keys from env vars
 * This is used when pushing to the cloud to avoid sending SDK keys
 */
export declare function filterAgentuitySdkKeys(vars: EnvVars): EnvVars;
/**
 * Split env vars into env and secrets based on key names
 * Convention: Keys ending with _SECRET, _KEY, _TOKEN, _PASSWORD are secrets
 * Also treats DATABASE_URL as a secret since it contains credentials
 */
export declare function splitEnvAndSecrets(vars: EnvVars): {
    env: EnvVars;
    secrets: EnvVars;
};
/**
 * Mask a secret value for display
 * - For URLs with credentials: parses and masks username/password partially
 * - For long strings: shows more context (first 6 and last 6 chars)
 * - For short strings: shows generic masking
 */
export declare function maskSecret(value: string): string;
/**
 * Detect if a key or value looks like it should be a secret
 */
export declare function looksLikeSecret(key: string, value: string): boolean;
/**
 * Add resource environment variables to the .env file.
 * Merges with existing variables, overwriting any that already exist.
 */
export declare function addResourceEnvVars(dir: string, envVars: EnvVars): Promise<void>;
/**
 * Remove resource environment variables from the .env file.
 * Only removes the specified keys if they exist.
 */
export declare function removeResourceEnvVars(dir: string, keys: string[]): Promise<void>;
/**
 * Get the value of a specific key from the .env file
 */
export declare function getEnvVar(dir: string, key: string): Promise<string | undefined>;
//# sourceMappingURL=env-util.d.ts.map