/**
 * Shared helpers for Agentuity Auth setup
 */
import type { Logger } from '../../../types';
import type { AuthData } from '../../../types';
/**
 * Database info returned from selection
 */
export interface DatabaseInfo {
    name: string;
    url: string;
}
/**
 * Select an existing database or create a new one
 */
export declare function selectOrCreateDatabase(options: {
    logger: Logger;
    auth: AuthData;
    orgId: string;
    region: string;
    existingUrl?: string;
}): Promise<DatabaseInfo>;
/**
 * Required auth dependencies
 */
export declare const AUTH_DEPENDENCIES: {
    readonly '@agentuity/auth': "latest";
    readonly 'better-auth': "^1.4.9";
    readonly 'drizzle-orm': "^0.45.0";
    readonly 'drizzle-kit': "^0.31.0";
};
/**
 * Check and install auth dependencies
 */
export declare function ensureAuthDependencies(options: {
    projectDir: string;
    logger: Logger;
}): Promise<boolean>;
/**
 * ORM setup type detected in a project.
 */
export type OrmSetup = 'drizzle' | 'prisma' | 'none';
/**
 * Get the directory for generated SQL files.
 * Uses src/generated/ if it exists, otherwise falls back to project root.
 */
export declare function getGeneratedSqlDir(projectDir: string): Promise<string>;
/**
 * Detect existing ORM setup in project.
 * TODO: This is probably not 100% accurate. Drizzle config could be in all sorts of places in a repo.
 */
export declare function detectOrmSetup(projectDir: string): Promise<OrmSetup>;
/**
 * Generate auth schema SQL using drizzle-kit export.
 *
 * This generates SQL DDL statements from the @agentuity/auth Drizzle schema
 * without needing a database connection.
 *
 * @param logger - Logger instance
 * @param projectDir - Project directory (must have @agentuity/auth installed)
 * @returns SQL DDL statements for auth tables
 */
export declare function generateAuthSchemaSql(logger: Logger, projectDir: string): Promise<string>;
/**
 * Split SQL into individual statements for sequential execution
 * The dbQuery API only supports single statements
 */
export declare function splitSqlStatements(sql: string): string[];
/**
 * Run auth migrations against a database.
 *
 * @param options.sql - SQL to execute (from generateAuthSchemaSql or custom)
 */
export declare function runAuthMigrations(options: {
    logger: Logger;
    auth: AuthData;
    orgId: string;
    region: string;
    databaseName: string;
    sql: string;
}): Promise<void>;
/**
 * Generate the auth.ts file content
 */
export declare function generateAuthFileContent(): string;
/**
 * Print integration examples to the console
 */
export declare function printIntegrationExamples(): void;
//# sourceMappingURL=shared.d.ts.map