import { z } from 'zod';
import { APIResponseSchema } from '../api';
import { SessionSchema } from './list';
import { SessionResponseError } from './util';
const _SessionGetRequestSchema = z.object({
    id: z.string().describe('the session id'),
});
const EvalRunSchema = z.object({
    id: z.string().describe('eval run id'),
    created_at: z.string().describe('creation timestamp'),
    eval_id: z.string().describe('evaluation id'),
    pending: z.boolean().describe('pending status'),
    success: z.boolean().describe('success status'),
    error: z.string().nullable().describe('error message'),
    result: z.string().nullable().describe('result JSON'),
});
const SpanNodeSchema = z.lazy(() => z.object({
    id: z.string().describe('span ID'),
    duration: z.number().describe('duration in milliseconds'),
    operation: z.string().describe('operation name'),
    attributes: z.record(z.string(), z.unknown()).describe('span attributes'),
    children: z.array(SpanNodeSchema).optional().describe('child spans'),
    error: z.string().optional().describe('error message'),
}));
const RouteInfoSchema = z
    .object({
    id: z.string().describe('route id'),
    method: z.string().describe('HTTP method'),
    path: z.string().describe('route path'),
})
    .nullable();
const AgentInfoSchema = z.object({
    name: z.string().describe('agent name'),
    identifier: z.string().describe('agent identifier'),
});
const EnrichedSessionDataSchema = z.object({
    session: SessionSchema,
    agents: z.array(AgentInfoSchema).describe('resolved agents'),
    eval_runs: z.array(EvalRunSchema).describe('eval runs for this session'),
    route: RouteInfoSchema.describe('route information'),
});
const SessionGetResponseSchema = APIResponseSchema(EnrichedSessionDataSchema);
export async function sessionGet(client, request) {
    const resp = await client.request('GET', `/session/2025-03-17/${request.id}`, SessionGetResponseSchema);
    if (resp.success) {
        return {
            session: resp.data.session,
            agents: resp.data.agents,
            evalRuns: resp.data.eval_runs,
            timeline: resp.data.session.timeline ?? null,
            route: resp.data.route,
        };
    }
    throw new SessionResponseError({ message: resp.message });
}
//# sourceMappingURL=get.js.map