import { z } from 'zod';
import { APIResponseSchema, APIResponseSchemaNoData } from '../api';
import { SandboxResponseError, API_VERSION } from './util';
const SnapshotFileInfoSchema = z
    .object({
    path: z.string().describe('File path within the snapshot'),
    size: z.number().describe('File size in bytes'),
})
    .describe('Information about a file in a snapshot');
const SnapshotInfoSchema = z
    .object({
    snapshotId: z.string().describe('Unique identifier for the snapshot'),
    tag: z.string().nullable().optional().describe('User-defined tag for the snapshot'),
    sizeBytes: z.number().describe('Total size of the snapshot in bytes'),
    fileCount: z.number().describe('Number of files in the snapshot'),
    parentSnapshotId: z
        .string()
        .nullable()
        .optional()
        .describe('ID of the parent snapshot (for incremental snapshots)'),
    createdAt: z.string().describe('ISO timestamp when the snapshot was created'),
    downloadUrl: z.string().optional().describe('URL to download the snapshot archive'),
    files: z.array(SnapshotFileInfoSchema).optional().describe('List of files in the snapshot'),
})
    .describe('Detailed information about a snapshot');
const SnapshotCreateResponseSchema = APIResponseSchema(SnapshotInfoSchema);
const SnapshotGetResponseSchema = APIResponseSchema(SnapshotInfoSchema);
const SnapshotListDataSchema = z
    .object({
    snapshots: z.array(SnapshotInfoSchema).describe('List of snapshot entries'),
    total: z.number().describe('Total number of snapshots matching the query'),
})
    .describe('Paginated list of snapshots');
const SnapshotListResponseSchema = APIResponseSchema(SnapshotListDataSchema);
const SnapshotDeleteResponseSchema = APIResponseSchemaNoData();
function buildQueryString(params) {
    const query = new URLSearchParams();
    for (const [key, value] of Object.entries(params)) {
        if (value !== undefined) {
            query.set(key, String(value));
        }
    }
    const str = query.toString();
    return str ? `?${str}` : '';
}
/**
 * Creates a snapshot of a sandbox's current state.
 *
 * @param client - The API client to use for the request
 * @param params - Parameters including sandbox ID and optional tag
 * @returns The created snapshot information
 * @throws {SandboxResponseError} If the snapshot creation fails
 */
export async function snapshotCreate(client, params) {
    const { sandboxId, tag, orgId } = params;
    const queryString = buildQueryString({ orgId });
    const url = `/sandbox/${API_VERSION}/${sandboxId}/snapshot${queryString}`;
    const body = {};
    if (tag) {
        body.tag = tag;
    }
    const resp = await client.post(url, body, SnapshotCreateResponseSchema);
    if (resp.success) {
        return resp.data;
    }
    throw new SandboxResponseError({ message: resp.message });
}
/**
 * Retrieves detailed information about a specific snapshot.
 *
 * @param client - The API client to use for the request
 * @param params - Parameters including the snapshot ID
 * @returns Snapshot information including files and download URL
 * @throws {SandboxResponseError} If the snapshot is not found or request fails
 */
export async function snapshotGet(client, params) {
    const { snapshotId, orgId } = params;
    const queryString = buildQueryString({ orgId });
    const url = `/sandbox/${API_VERSION}/snapshots/${snapshotId}${queryString}`;
    const resp = await client.get(url, SnapshotGetResponseSchema);
    if (resp.success) {
        return resp.data;
    }
    throw new SandboxResponseError({ message: resp.message });
}
/**
 * Lists snapshots with optional filtering and pagination.
 *
 * @param client - The API client to use for the request
 * @param params - Optional parameters for filtering by sandbox and pagination
 * @returns Paginated list of snapshots with total count
 * @throws {SandboxResponseError} If the request fails
 */
export async function snapshotList(client, params = {}) {
    const { sandboxId, limit, offset, orgId } = params;
    const queryString = buildQueryString({ sandboxId, limit, offset, orgId });
    const url = `/sandbox/${API_VERSION}/snapshots${queryString}`;
    const resp = await client.get(url, SnapshotListResponseSchema);
    if (resp.success) {
        return resp.data;
    }
    throw new SandboxResponseError({ message: resp.message });
}
/**
 * Deletes a snapshot and releases its storage.
 *
 * @param client - The API client to use for the request
 * @param params - Parameters including the snapshot ID to delete
 * @throws {SandboxResponseError} If the snapshot is not found or deletion fails
 */
export async function snapshotDelete(client, params) {
    const { snapshotId, orgId } = params;
    const queryString = buildQueryString({ orgId });
    const url = `/sandbox/${API_VERSION}/snapshots/${snapshotId}${queryString}`;
    const resp = await client.delete(url, SnapshotDeleteResponseSchema);
    if (!resp.success) {
        throw new SandboxResponseError({ message: resp.message });
    }
}
/**
 * Updates or removes the tag on a snapshot.
 *
 * @param client - The API client to use for the request
 * @param params - Parameters including snapshot ID and new tag (or null to remove)
 * @returns Updated snapshot information
 * @throws {SandboxResponseError} If the snapshot is not found or update fails
 */
export async function snapshotTag(client, params) {
    const { snapshotId, tag, orgId } = params;
    const queryString = buildQueryString({ orgId });
    const url = `/sandbox/${API_VERSION}/snapshots/${snapshotId}${queryString}`;
    const resp = await client.patch(url, { tag }, SnapshotGetResponseSchema);
    if (resp.success) {
        return resp.data;
    }
    throw new SandboxResponseError({ message: resp.message });
}
//# sourceMappingURL=snapshot.js.map