import { APIClient } from '../api';
import { sandboxCreate } from './create';
import { sandboxDestroy } from './destroy';
import { sandboxGet } from './get';
import { sandboxExecute } from './execute';
import { sandboxWriteFiles, sandboxReadFile } from './files';
import { executionGet } from './execution';
import { ConsoleLogger } from '../../logger';
import { getServiceUrls } from '../../config';
const POLL_INTERVAL_MS = 100;
const MAX_POLL_TIME_MS = 300000; // 5 minutes
/**
 * Poll for execution completion
 */
async function waitForExecution(client, executionId, orgId, signal) {
    const startTime = Date.now();
    while (Date.now() - startTime < MAX_POLL_TIME_MS) {
        if (signal?.aborted) {
            throw new DOMException('The operation was aborted.', 'AbortError');
        }
        const info = await executionGet(client, { executionId, orgId });
        if (info.status === 'completed' ||
            info.status === 'failed' ||
            info.status === 'timeout' ||
            info.status === 'cancelled') {
            return info;
        }
        await new Promise((resolve, reject) => {
            const timeoutId = setTimeout(resolve, POLL_INTERVAL_MS);
            signal?.addEventListener('abort', () => {
                clearTimeout(timeoutId);
                reject(new DOMException('The operation was aborted.', 'AbortError'));
            }, { once: true });
        });
    }
    throw new Error(`Execution ${executionId} timed out waiting for completion`);
}
/**
 * Pipes a remote stream URL to a local writable stream
 */
async function pipeStreamToWritable(streamUrl, writable) {
    const response = await fetch(streamUrl);
    if (!response.ok) {
        throw new Error(`Failed to fetch stream: ${response.status} ${response.statusText}`);
    }
    if (!response.body) {
        return;
    }
    const reader = response.body.getReader();
    try {
        while (true) {
            const { done, value } = await reader.read();
            if (done)
                break;
            if (value) {
                writable.write(value);
            }
        }
    }
    finally {
        reader.releaseLock();
    }
}
/**
 * Convenience client for sandbox operations.
 *
 * @example
 * ```typescript
 * const client = new SandboxClient();
 * const sandbox = await client.create();
 * const result = await sandbox.execute({ command: ['echo', 'hello'] });
 * await sandbox.destroy();
 * ```
 */
export class SandboxClient {
    #client;
    #orgId;
    constructor(options = {}) {
        const apiKey = options.apiKey || process.env.AGENTUITY_SDK_KEY || process.env.AGENTUITY_CLI_KEY;
        const region = process.env.AGENTUITY_REGION ?? 'usc';
        const serviceUrls = getServiceUrls(region);
        const url = options.url ||
            process.env.AGENTUITY_SANDBOX_URL ||
            process.env.AGENTUITY_CATALYST_URL ||
            process.env.AGENTUITY_TRANSPORT_URL ||
            serviceUrls.sandbox;
        const logger = options.logger ?? new ConsoleLogger('warn');
        this.#client = new APIClient(url, logger, apiKey ?? '', {});
        this.#orgId = options.orgId;
    }
    /**
     * Create a new sandbox instance
     *
     * @param options - Optional sandbox configuration
     * @returns A sandbox instance with execute and destroy methods
     */
    async create(options) {
        const response = await sandboxCreate(this.#client, {
            options,
            orgId: this.#orgId,
        });
        const sandboxId = response.sandboxId;
        const client = this.#client;
        const orgId = this.#orgId;
        return {
            id: sandboxId,
            status: response.status,
            stdoutStreamUrl: response.stdoutStreamUrl,
            stderrStreamUrl: response.stderrStreamUrl,
            async execute(executeOptions) {
                const { pipe, ...coreOptions } = executeOptions;
                const initialResult = await sandboxExecute(client, {
                    sandboxId,
                    options: coreOptions,
                    orgId,
                    signal: coreOptions.signal,
                });
                // If pipe options provided, stream the output to the writable streams
                if (pipe) {
                    const streamPromises = [];
                    if (pipe.stdout && initialResult.stdoutStreamUrl) {
                        streamPromises.push(pipeStreamToWritable(initialResult.stdoutStreamUrl, pipe.stdout));
                    }
                    if (pipe.stderr && initialResult.stderrStreamUrl) {
                        streamPromises.push(pipeStreamToWritable(initialResult.stderrStreamUrl, pipe.stderr));
                    }
                    // Wait for all streams to complete
                    if (streamPromises.length > 0) {
                        await Promise.all(streamPromises);
                    }
                }
                // Wait for execution to complete and get final result with exit code
                const finalResult = await waitForExecution(client, initialResult.executionId, orgId, coreOptions.signal);
                return {
                    executionId: finalResult.executionId,
                    status: finalResult.status,
                    exitCode: finalResult.exitCode,
                    durationMs: finalResult.durationMs,
                    stdoutStreamUrl: initialResult.stdoutStreamUrl,
                    stderrStreamUrl: initialResult.stderrStreamUrl,
                };
            },
            async get() {
                return sandboxGet(client, { sandboxId, orgId });
            },
            async destroy() {
                return sandboxDestroy(client, { sandboxId, orgId });
            },
        };
    }
    /**
     * Get sandbox information by ID
     *
     * @param sandboxId - The sandbox ID
     * @returns Sandbox information
     */
    async get(sandboxId) {
        return sandboxGet(this.#client, { sandboxId, orgId: this.#orgId });
    }
    /**
     * Destroy a sandbox by ID
     *
     * @param sandboxId - The sandbox ID to destroy
     */
    async destroy(sandboxId) {
        return sandboxDestroy(this.#client, { sandboxId, orgId: this.#orgId });
    }
    /**
     * Write files to a sandbox workspace
     *
     * @param sandboxId - The sandbox ID
     * @param files - Array of files to write with path and content
     * @param signal - Optional AbortSignal to cancel the operation
     * @returns The number of files written
     */
    async writeFiles(sandboxId, files, signal) {
        const result = await sandboxWriteFiles(this.#client, {
            sandboxId,
            files,
            orgId: this.#orgId,
            signal,
        });
        return result.filesWritten;
    }
    /**
     * Read a file from a sandbox workspace
     *
     * @param sandboxId - The sandbox ID
     * @param path - Path to the file relative to the sandbox workspace
     * @param signal - Optional AbortSignal to cancel the operation
     * @returns A ReadableStream of the file contents
     */
    async readFile(sandboxId, path, signal) {
        return sandboxReadFile(this.#client, {
            sandboxId,
            path,
            orgId: this.#orgId,
            signal,
        });
    }
}
//# sourceMappingURL=client.js.map