import type { Logger, SandboxCreateOptions, SandboxInfo, ExecuteOptions as CoreExecuteOptions, Execution, FileToWrite } from '@agentuity/core';
import type { Writable } from 'node:stream';
import { type SandboxCreateResponse } from './create';
/**
 * Extended execute options that support piping output to writable streams
 */
export interface ExecuteOptions extends CoreExecuteOptions {
    /**
     * Pipe stdout/stderr to writable streams (e.g., process.stdout)
     */
    pipe?: {
        stdout?: Writable;
        stderr?: Writable;
    };
}
export interface SandboxClientOptions {
    /**
     * API key for authentication.
     * Defaults to process.env.AGENTUITY_SDK_KEY || process.env.AGENTUITY_CLI_KEY
     */
    apiKey?: string;
    /**
     * Base URL for the sandbox API.
     * Defaults to process.env.AGENTUITY_SANDBOX_URL ||
     *   process.env.AGENTUITY_CATALYST_URL ||
     *   process.env.AGENTUITY_TRANSPORT_URL ||
     *   regional catalyst URL
     */
    url?: string;
    /**
     * Organization ID for multi-tenant operations
     */
    orgId?: string;
    /**
     * Custom logger instance
     */
    logger?: Logger;
}
/**
 * A sandbox instance returned by SandboxClient.create()
 */
export interface SandboxInstance {
    /**
     * Unique sandbox identifier
     */
    id: string;
    /**
     * Sandbox status at creation time
     */
    status: SandboxCreateResponse['status'];
    /**
     * URL to stream stdout output
     */
    stdoutStreamUrl?: string;
    /**
     * URL to stream stderr output
     */
    stderrStreamUrl?: string;
    /**
     * Execute a command in the sandbox
     */
    execute(options: ExecuteOptions): Promise<Execution>;
    /**
     * Get current sandbox information
     */
    get(): Promise<SandboxInfo>;
    /**
     * Destroy the sandbox and release all resources
     */
    destroy(): Promise<void>;
}
/**
 * Convenience client for sandbox operations.
 *
 * @example
 * ```typescript
 * const client = new SandboxClient();
 * const sandbox = await client.create();
 * const result = await sandbox.execute({ command: ['echo', 'hello'] });
 * await sandbox.destroy();
 * ```
 */
export declare class SandboxClient {
    #private;
    constructor(options?: SandboxClientOptions);
    /**
     * Create a new sandbox instance
     *
     * @param options - Optional sandbox configuration
     * @returns A sandbox instance with execute and destroy methods
     */
    create(options?: SandboxCreateOptions): Promise<SandboxInstance>;
    /**
     * Get sandbox information by ID
     *
     * @param sandboxId - The sandbox ID
     * @returns Sandbox information
     */
    get(sandboxId: string): Promise<SandboxInfo>;
    /**
     * Destroy a sandbox by ID
     *
     * @param sandboxId - The sandbox ID to destroy
     */
    destroy(sandboxId: string): Promise<void>;
    /**
     * Write files to a sandbox workspace
     *
     * @param sandboxId - The sandbox ID
     * @param files - Array of files to write with path and content
     * @param signal - Optional AbortSignal to cancel the operation
     * @returns The number of files written
     */
    writeFiles(sandboxId: string, files: FileToWrite[], signal?: AbortSignal): Promise<number>;
    /**
     * Read a file from a sandbox workspace
     *
     * @param sandboxId - The sandbox ID
     * @param path - Path to the file relative to the sandbox workspace
     * @param signal - Optional AbortSignal to cancel the operation
     * @returns A ReadableStream of the file contents
     */
    readFile(sandboxId: string, path: string, signal?: AbortSignal): Promise<ReadableStream<Uint8Array>>;
}
//# sourceMappingURL=client.d.ts.map