import { stream as honoStream } from 'hono/streaming';
import { getAgentAsyncLocalStorage } from '../_context';
import { STREAM_DONE_PROMISE_KEY, IS_STREAMING_RESPONSE_KEY } from './sse';
/**
 * Creates a streaming middleware for returning ReadableStream responses.
 *
 * Use with router.post() (or any HTTP method) to create a streaming endpoint:
 *
 * @example
 * ```typescript
 * import { createRouter, stream } from '@agentuity/runtime';
 *
 * const router = createRouter();
 *
 * router.post('/events', stream((c) => {
 *   return new ReadableStream({
 *     start(controller) {
 *       controller.enqueue('event 1\n');
 *       controller.enqueue('event 2\n');
 *       controller.close();
 *     }
 *   });
 * }));
 * ```
 *
 * @example
 * ```typescript
 * // Async stream with data from request body
 * router.post('/process', stream(async (c) => {
 *   const body = await c.req.json();
 *
 *   return new ReadableStream({
 *     async start(controller) {
 *       for (const item of body.items) {
 *         controller.enqueue(`Processing: ${item}\n`);
 *         await new Promise(r => setTimeout(r, 100));
 *       }
 *       controller.close();
 *     }
 *   });
 * }));
 * ```
 *
 * @param handler - Handler function returning a ReadableStream
 * @returns Hono handler for streaming response
 */
export function stream(handler) {
    return (c) => {
        const asyncLocalStorage = getAgentAsyncLocalStorage();
        const capturedContext = asyncLocalStorage.getStore();
        // Track stream completion for deferred session/thread saving
        // This promise resolves when the stream completes (pipe finishes or errors)
        let resolveDone;
        let rejectDone;
        const donePromise = new Promise((resolve, reject) => {
            resolveDone = resolve;
            rejectDone = reject;
        });
        // Idempotent function to mark stream as completed
        let isDone = false;
        const markDone = (error) => {
            if (isDone)
                return;
            isDone = true;
            if (error && rejectDone) {
                rejectDone(error);
            }
            else if (resolveDone) {
                resolveDone();
            }
        };
        // Expose completion tracking to middleware
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        c.set(STREAM_DONE_PROMISE_KEY, donePromise);
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        c.set(IS_STREAMING_RESPONSE_KEY, true);
        c.header('Content-Type', 'application/octet-stream');
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        return honoStream(c, async (s) => {
            const runInContext = async () => {
                try {
                    let streamResult = handler(c);
                    if (streamResult instanceof Promise) {
                        streamResult = await streamResult;
                    }
                    await s.pipe(streamResult);
                    // Stream completed successfully
                    markDone();
                }
                catch (err) {
                    c.var.logger?.error('Stream error:', err);
                    markDone(err);
                    throw err;
                }
            };
            if (capturedContext) {
                await asyncLocalStorage.run(capturedContext, runInContext);
            }
            else {
                await runInContext();
            }
        });
    };
}
//# sourceMappingURL=stream.js.map