/**
 * Build metadata utilities
 * Provides cached access to agentuity.metadata.json
 */
export interface BuildMetadataAgent {
    filename: string;
    id: string;
    agentId: string;
    version: string;
    name: string;
    description?: string;
    projectId?: string;
    schema?: {
        input?: string;
        output?: string;
    };
    evals?: Array<{
        filename: string;
        id: string;
        identifier: string;
        name: string;
        version: string;
        description?: string;
        agentIdentifier?: string;
        projectId?: string;
    }>;
}
export interface BuildMetadataRoute {
    id: string;
    filename: string;
    path: string;
    method: 'get' | 'post' | 'put' | 'delete' | 'patch';
    version: string;
    type?: string;
    agentIds?: string[];
    config?: Record<string, unknown>;
    schema?: {
        input?: string;
        output?: string;
    };
}
export interface BuildMetadata {
    routes: BuildMetadataRoute[];
    agents: BuildMetadataAgent[];
    assets?: string[];
    project: {
        id: string;
        name: string;
        version?: string;
        description?: string;
        keywords?: string[];
        orgId?: string;
    };
    deployment: {
        id: string;
        date: string;
        build: {
            bun: string;
            agentuity: string;
            arch: string;
            platform: string;
        };
        git?: {
            branch?: string;
            repo?: string;
            provider?: string;
            tags?: string[];
            commit?: string;
            message?: string;
        };
    };
}
/**
 * Get the path to agentuity.metadata.json
 *
 * Checks multiple locations to support both dev and production:
 * - Production: cwd is .agentuity/, file is at cwd/agentuity.metadata.json
 * - Dev: cwd is project root, file is at cwd/.agentuity/agentuity.metadata.json
 */
export declare function getMetadataPath(): string;
/**
 * Load and cache the build metadata from agentuity.metadata.json
 * Returns undefined if the file doesn't exist or can't be parsed
 */
export declare function loadBuildMetadata(): BuildMetadata | undefined;
export type BuildMetadataEval = NonNullable<BuildMetadataAgent['evals']>[number];
/**
 * Look up agent metadata by name
 */
export declare function getAgentMetadataByName(agentName: string): BuildMetadataAgent | undefined;
/**
 * Look up agent metadata by agentId
 */
export declare function getAgentMetadataByAgentId(agentId: string): BuildMetadataAgent | undefined;
/**
 * Look up eval metadata by agent name and eval name
 */
export declare function getEvalMetadata(agentName: string, evalName: string): BuildMetadataEval | undefined;
/**
 * Look up eval metadata by agentId and eval name
 */
export declare function getEvalMetadataByAgentId(agentId: string, evalName: string): BuildMetadataEval | undefined;
/**
 * Check if metadata file exists (uses cache)
 */
export declare function hasMetadata(): boolean;
/**
 * Import all agents from metadata filenames to ensure they're registered.
 * This is needed so that runtime schemas are available for JSON schema generation.
 * Safe to call multiple times - will only import once.
 */
export declare function ensureAgentsImported(): Promise<void>;
/**
 * Clear the metadata cache (useful for testing or hot reload)
 */
export declare function clearMetadataCache(): void;
//# sourceMappingURL=_metadata.d.ts.map