import { z } from 'zod';
import { createCommand } from '../../../types';
import * as tui from '../../../tui';
import { createSandboxClient } from './util';
import { getCommand } from '../../../command-prefix';
import { sandboxGet } from '@agentuity/server';
const SandboxResourcesSchema = z.object({
    memory: z.string().optional().describe('Memory limit (e.g., "512Mi", "1Gi")'),
    cpu: z.string().optional().describe('CPU limit (e.g., "500m", "1000m")'),
    disk: z.string().optional().describe('Disk limit (e.g., "1Gi", "10Gi")'),
});
const SandboxGetResponseSchema = z.object({
    sandboxId: z.string().describe('Sandbox ID'),
    status: z.string().describe('Current status'),
    createdAt: z.string().describe('Creation timestamp'),
    region: z.string().optional().describe('Region where sandbox is running'),
    snapshotId: z.string().optional().describe('Snapshot ID sandbox was created from'),
    snapshotTag: z.string().optional().describe('Snapshot tag sandbox was created from'),
    executions: z.number().describe('Number of executions'),
    stdoutStreamUrl: z.string().optional().describe('URL to stdout output stream'),
    stderrStreamUrl: z.string().optional().describe('URL to stderr output stream'),
    dependencies: z.array(z.string()).optional().describe('Apt packages installed'),
    metadata: z.record(z.string(), z.unknown()).optional().describe('User-defined metadata'),
    resources: SandboxResourcesSchema.optional().describe('Resource limits'),
});
export const getSubcommand = createCommand({
    name: 'get',
    aliases: ['info', 'show'],
    description: 'Get information about a sandbox',
    tags: ['read-only', 'fast', 'requires-auth'],
    requires: { auth: true, region: true, org: true },
    idempotent: true,
    examples: [
        {
            command: getCommand('cloud sandbox get abc123'),
            description: 'Get sandbox information',
        },
    ],
    schema: {
        args: z.object({
            sandboxId: z.string().describe('Sandbox ID'),
        }),
        response: SandboxGetResponseSchema,
    },
    async handler(ctx) {
        const { args, options, auth, region, logger, orgId } = ctx;
        const client = createSandboxClient(logger, auth, region);
        const result = await sandboxGet(client, { sandboxId: args.sandboxId, orgId });
        if (!options.json) {
            const statusColor = result.status === 'running'
                ? tui.colorSuccess
                : result.status === 'idle'
                    ? tui.colorWarning
                    : result.status === 'failed'
                        ? tui.colorError
                        : tui.colorMuted;
            console.log(`${tui.muted('Sandbox:')}         ${tui.bold(result.sandboxId)}`);
            console.log(`${tui.muted('Status:')}          ${statusColor(result.status)}`);
            console.log(`${tui.muted('Created:')}         ${result.createdAt}`);
            if (result.region) {
                console.log(`${tui.muted('Region:')}          ${result.region}`);
            }
            if (result.snapshotId || result.snapshotTag) {
                const snapshotDisplay = result.snapshotTag
                    ? `${result.snapshotTag} ${tui.muted('(' + result.snapshotId + ')')}`
                    : result.snapshotId;
                console.log(`${tui.muted('Snapshot:')}        ${snapshotDisplay}`);
            }
            console.log(`${tui.muted('Executions:')}      ${result.executions}`);
            if (result.stdoutStreamUrl &&
                result.stderrStreamUrl &&
                result.stdoutStreamUrl === result.stderrStreamUrl) {
                console.log(`${tui.muted('Stream:')}          ${tui.link(result.stdoutStreamUrl)}`);
            }
            else {
                if (result.stdoutStreamUrl) {
                    console.log(`${tui.muted('Stream (stdout):')} ${tui.link(result.stdoutStreamUrl)}`);
                }
                if (result.stderrStreamUrl) {
                    console.log(`${tui.muted('Stream (stderr):')} ${tui.link(result.stderrStreamUrl)}`);
                }
            }
            if (result.dependencies && result.dependencies.length > 0) {
                console.log(`${tui.muted('Dependencies:')}    ${result.dependencies.join(', ')}`);
            }
            if (result.resources) {
                const resourceParts = [];
                if (result.resources.memory)
                    resourceParts.push(`memory=${result.resources.memory}`);
                if (result.resources.cpu)
                    resourceParts.push(`cpu=${result.resources.cpu}`);
                if (result.resources.disk)
                    resourceParts.push(`disk=${result.resources.disk}`);
                if (resourceParts.length > 0) {
                    console.log(`${tui.muted('Resources:')}       ${resourceParts.join(', ')}`);
                }
            }
            if (result.metadata && Object.keys(result.metadata).length > 0) {
                console.log(`${tui.muted('Metadata:')}        ${JSON.stringify(result.metadata)}`);
            }
        }
        return {
            sandboxId: result.sandboxId,
            status: result.status,
            createdAt: result.createdAt,
            region: result.region,
            snapshotId: result.snapshotId,
            snapshotTag: result.snapshotTag,
            executions: result.executions,
            stdoutStreamUrl: result.stdoutStreamUrl,
            stderrStreamUrl: result.stderrStreamUrl,
            dependencies: result.dependencies,
            metadata: result.metadata,
            resources: result.resources,
        };
    },
});
export default getSubcommand;
//# sourceMappingURL=get.js.map