/**
 * CLI-specific API client wrapper
 *
 * Re-exports from @agentuity/server with CLI-specific configuration
 */
import { getVersion, getRevision } from './version';
import { APIClient as BaseAPIClient, getAPIBaseURL as baseGetAPIBaseURL, getAppBaseURL as baseGetAppBaseURL, } from '@agentuity/server';
export function getUserAgent(config) {
    // If we're skipping version check, send "dev" to signal the server to skip too
    let version = getVersion();
    if (shouldSkipVersionCheck(config)) {
        version = 'dev';
    }
    const revision = getRevision();
    return `Agentuity CLI/${version} (${revision})`;
}
function shouldSkipVersionCheck(config) {
    // Priority order:
    // 1. CLI flag (set via env var in cli.ts)
    // 2. Environment variable
    // 3. Config override
    // 4. Auto-detection (dev/0.0.x versions)
    // Skip if environment variable is set (includes CLI flag)
    if (process.env.AGENTUITY_SKIP_VERSION_CHECK === '1' ||
        process.env.AGENTUITY_SKIP_VERSION_CHECK === 'true') {
        return true;
    }
    // Check config overrides
    const overrides = config?.overrides;
    if (overrides?.skip_version_check === true) {
        return true;
    }
    // Skip if version is 'dev' or starts with '0.0.' (pre-release/local development)
    const version = getVersion();
    if (version === 'dev' || version.startsWith('0.0.')) {
        return true;
    }
    return false;
}
// CLI-specific wrapper around the base APIClient
export class APIClient extends BaseAPIClient {
    constructor(baseUrl, logger, apiKeyOrConfig, config) {
        const clientConfig = {
            skipVersionCheck: shouldSkipVersionCheck(typeof apiKeyOrConfig === 'string' ? config : apiKeyOrConfig),
            userAgent: getUserAgent(typeof apiKeyOrConfig === 'string' ? config : apiKeyOrConfig),
        };
        if (typeof apiKeyOrConfig === 'string') {
            super(baseUrl, logger, apiKeyOrConfig, clientConfig);
        }
        else {
            if (apiKeyOrConfig?.auth?.api_key) {
                super(baseUrl, logger, apiKeyOrConfig.auth.api_key, clientConfig);
            }
            else {
                super(baseUrl, logger, clientConfig);
            }
        }
    }
}
export function getAPIBaseURL(config) {
    const overrides = config?.overrides;
    return baseGetAPIBaseURL(config?.name, overrides);
}
export function getAppBaseURL(config) {
    const overrides = config?.overrides;
    return baseGetAppBaseURL(config?.name, overrides);
}
export function getGravityDevModeURL(region, config) {
    const overrides = config?.overrides;
    if (overrides?.gravity_url) {
        return overrides.gravity_url;
    }
    if (config?.name === 'local') {
        return 'grpc://gravity.agentuity.io:8443';
    }
    return `grpc://gravity-${region}.agentuity.cloud`;
}
//# sourceMappingURL=api.js.map