import { type Client, type ClientOptions, type RPCRouteRegistry } from '@agentuity/frontend';
export type { RPCRouteRegistry };
/**
 * Set the global base URL for RPC clients.
 * This is automatically called by setContext in Svelte apps.
 * @internal
 */
export declare function setGlobalBaseUrl(url: string): void;
/**
 * Get the global base URL for RPC clients.
 * Returns the configured base URL or falls back to window.location.origin.
 * @internal
 */
export declare function getGlobalBaseUrl(): string;
/**
 * Set the global auth header for RPC clients.
 * This is automatically called when auth state changes.
 * @internal
 */
export declare function setGlobalAuthHeader(authHeader: string | null): void;
/**
 * Get the global auth header for RPC clients.
 * Returns the current auth header or undefined if not set.
 * @internal
 */
export declare function getGlobalAuthHeader(): string | null | undefined;
/**
 * Create a type-safe RPC client for Svelte applications.
 *
 * This is a Svelte-specific wrapper around @agentuity/frontend's createClient that
 * automatically uses the baseUrl and auth headers from global state.
 *
 * @example
 * ```typescript
 * import { createClient } from '@agentuity/svelte';
 * import type { RPCRouteRegistry } from '@agentuity/svelte';
 *
 * const client = createClient<RPCRouteRegistry>();
 *
 * // Inside component
 * const result = await client.hello.post({ name: 'World' });
 * ```
 */
export declare function createClient<R>(options?: Omit<ClientOptions, 'baseUrl' | 'headers'> & {
    baseUrl?: string | (() => string);
    headers?: Record<string, string> | (() => Record<string, string>);
}, metadata?: unknown): Client<R>;
/**
 * Create a type-safe API client with optional configuration.
 *
 * This is the recommended way to create an API client in Svelte applications.
 * It automatically includes auth headers from global state and allows
 * custom headers to be passed.
 *
 * The generic type parameter defaults to RPCRouteRegistry which is augmented
 * by generated code, so you don't need to specify it manually.
 *
 * @example
 * ```typescript
 * import { createAPIClient } from '@agentuity/svelte';
 *
 * // Types are automatically inferred from generated routes
 * const api = createAPIClient();
 * await api.hello.post({ name: 'World' });
 *
 * // With custom headers
 * const api = createAPIClient({ headers: { 'X-Custom': 'value' } });
 * await api.hello.post({ name: 'World' });
 * ```
 */
export declare function createAPIClient<R = RPCRouteRegistry>(options?: Omit<ClientOptions, 'baseUrl' | 'headers'> & {
    baseUrl?: string | (() => string);
    headers?: Record<string, string> | (() => Record<string, string>);
}): Client<R>;
//# sourceMappingURL=client.d.ts.map