import { z } from 'zod';
import { APIResponseSchema } from '../api';
import { RegionResponseError } from './util';
const ResourceListResponse = z.object({
    s3: z.array(z.object({
        bucket_name: z.string().describe('the S3 bucket name'),
        access_key: z.string().nullable().optional().describe('the S3 access key'),
        secret_key: z.string().nullable().optional().describe('the S3 secret key'),
        region: z.string().nullable().optional().describe('the S3 region'),
        endpoint: z.string().nullable().optional().describe('the S3 endpoint'),
        cname: z.string().nullable().optional().describe('the S3 CNAME'),
        env: z.record(z.string(), z.string()).describe('environment variables for the resource'),
        bucket_type: z.string().describe('the bucket type (user or snapshots)'),
        internal: z.boolean().describe('whether this is a system-managed bucket'),
        description: z
            .string()
            .nullable()
            .optional()
            .describe('optional description of the bucket'),
    })),
    db: z.array(z.object({
        name: z.string().describe('the database name'),
        username: z.string().nullable().optional().describe('the database username'),
        password: z.string().nullable().optional().describe('the database password'),
        url: z.string().nullable().optional().describe('the full database connection URL'),
        env: z.record(z.string(), z.string()).describe('environment variables for the resource'),
        internal: z.boolean().describe('whether this is a system-managed database (KV/Vector/Queue)'),
    })),
    redis: z
        .object({
        url: z.string().describe('the Redis connection URL'),
    })
        .optional(),
});
const ResourceListResponseSchema = APIResponseSchema(ResourceListResponse);
/**
 * List all resources for a given organization and region
 *
 * @param client
 * @param orgId
 * @param region
 * @returns
 */
export async function listResources(client, orgId, region) {
    const resp = await client.request('GET', `/resource/2025-03-17/${orgId}/${region}`, ResourceListResponseSchema);
    if (resp.success) {
        return resp.data;
    }
    throw new RegionResponseError({ message: resp.message });
}
//# sourceMappingURL=resources.js.map