import { checkAuth } from '../../services/auth';
const CLOUD_TOOL_PREFIXES = [
    'agentuity.kv',
    'agentuity.storage',
    'agentuity.vector',
    'agentuity.sandbox',
];
/**
 * Get the Agentuity profile to use for CLI commands.
 * Defaults to 'production' for safety, but can be overridden via AGENTUITY_CODER_PROFILE.
 */
function getCoderProfile() {
    const profile = process.env.AGENTUITY_CODER_PROFILE?.trim();
    return profile || 'production';
}
/** Cloud service detection for bash commands */
const CLOUD_SERVICES = {
    'agentuity cloud kv': { name: 'KV Storage', emoji: '🗄️' },
    'agentuity cloud storage': { name: 'Object Storage', emoji: '📦' },
    'agentuity cloud vector': { name: 'Vector Search', emoji: '🔍' },
    'agentuity cloud sandbox': { name: 'Sandbox', emoji: '🏖️' },
    'agentuity cloud db': { name: 'Postgres', emoji: '🐘' },
    'agentuity cloud ssh': { name: 'SSH', emoji: '🔐' },
    'agentuity cloud scp': { name: 'File Transfer', emoji: '📤' },
};
export function createToolHooks(ctx, config) {
    const blockedCommands = config.blockedCommands ?? [];
    return {
        async before(input, output) {
            const toolName = extractToolName(input);
            if (!toolName)
                return;
            // Check MCP cloud tools
            if (isCloudTool(toolName)) {
                const authResult = await checkAuth();
                if (!authResult.ok) {
                    const out = output;
                    out.error = authResult.error;
                }
            }
            // Detect bash commands using agentuity CLI
            if (toolName === 'bash') {
                const command = extractBashCommand(input);
                if (command?.includes('agentuity')) {
                    // Security: Block sensitive commands
                    const blockedPattern = isBlockedCommand(command, blockedCommands);
                    if (blockedPattern) {
                        const out = output;
                        out.blocked = true;
                        out.error = `🚫 Blocked: "${blockedPattern}" commands are not allowed for security reasons.`;
                        ctx.client.app.log({
                            body: {
                                service: 'agentuity-coder',
                                level: 'warn',
                                message: `Blocked command pattern: ${blockedPattern}`,
                                extra: { command },
                            },
                        });
                        return;
                    }
                    // Inject AGENTUITY_PROFILE environment variable
                    const profile = getCoderProfile();
                    let modifiedCommand;
                    // Check if AGENTUITY_PROFILE already exists (anywhere in the command)
                    if (/AGENTUITY_PROFILE=\S+/.test(command)) {
                        // Replace all existing AGENTUITY_PROFILE occurrences to enforce our profile
                        modifiedCommand = command.replace(/AGENTUITY_PROFILE=\S+/g, `AGENTUITY_PROFILE=${profile}`);
                    }
                    else {
                        // Prepend AGENTUITY_PROFILE
                        modifiedCommand = `AGENTUITY_PROFILE=${profile} ${command}`;
                    }
                    setBashCommand(input, modifiedCommand);
                    // Show toast for cloud service usage
                    const service = detectCloudService(command);
                    if (service) {
                        try {
                            ctx.client.tui.showToast({
                                body: {
                                    message: `${service.emoji} Agentuity ${service.name}`,
                                    variant: 'info',
                                },
                            });
                        }
                        catch {
                            // Toast may not be available
                        }
                    }
                }
            }
        },
        async after(_input, _output) { },
    };
}
function extractToolName(input) {
    if (typeof input === 'object' && input !== null && 'tool' in input) {
        return input.tool;
    }
    return undefined;
}
function extractBashCommand(input) {
    if (typeof input !== 'object' || input === null)
        return undefined;
    const inp = input;
    // Try different possible arg structures
    if (typeof inp.command === 'string')
        return inp.command;
    if (typeof inp.args === 'object' && inp.args !== null) {
        const args = inp.args;
        if (typeof args.command === 'string')
            return args.command;
    }
    return undefined;
}
/**
 * Set the bash command in the input object.
 * Handles both direct command property and args.command structures.
 */
function setBashCommand(input, command) {
    if (typeof input !== 'object' || input === null)
        return;
    const inp = input;
    // Update the command in the same location it was found
    if (typeof inp.command === 'string') {
        inp.command = command;
    }
    else if (typeof inp.args === 'object' && inp.args !== null) {
        const args = inp.args;
        if (typeof args.command === 'string') {
            args.command = command;
        }
    }
}
function detectCloudService(command) {
    for (const [pattern, service] of Object.entries(CLOUD_SERVICES)) {
        if (command.includes(pattern)) {
            return service;
        }
    }
    return null;
}
function isCloudTool(toolName) {
    return CLOUD_TOOL_PREFIXES.some((prefix) => toolName.startsWith(prefix));
}
/** Check if a command matches any blocked pattern, returns the matched pattern or null */
function isBlockedCommand(command, blockedPatterns) {
    for (const pattern of blockedPatterns) {
        if (command.includes(pattern)) {
            return pattern;
        }
    }
    return null;
}
//# sourceMappingURL=tools.js.map