import { buildUrl, toServiceException } from './_util';
import { safeStringify } from '../json';
import { StructuredError } from '../error';
/**
 * Minimum TTL value in seconds (1 minute)
 */
export const VECTOR_MIN_TTL_SECONDS = 60;
/**
 * Maximum TTL value in seconds (90 days)
 */
export const VECTOR_MAX_TTL_SECONDS = 7776000;
/**
 * Default TTL value in seconds (30 days) - used when no TTL is specified
 */
export const VECTOR_DEFAULT_TTL_SECONDS = 2592000;
const VectorStorageNameRequiredError = StructuredError('VectorStorageNameRequiredError', 'Vector storage name is required and must be a non-empty string');
const VectorStorageKeyRequiredError = StructuredError('VectorStorageKeyRequiredError', 'Vector storage key is required and must be a non-empty string');
const VectorStorageKeysError = StructuredError('VectorStorageKeysError', 'Vector storage requires all keys to be non-empty strings');
const VectorStorageQueryError = StructuredError('VectorStorageQueryError', 'Vector storage query property is required and must be a non-empty string');
const VectorStorageLimitError = StructuredError('VectorStorageLimitError', 'Vector storage limit property must be positive number');
const VectorStorageSimilarityError = StructuredError('VectorStorageSimilarityError', 'Vector storage similarity property must be a number between 0.0 and 1.0');
const VectorStorageMetadataError = StructuredError('VectorStorageMetadataError', 'Vector storage metadata property must be a valid object');
const VectorStorageDocumentRequiredError = StructuredError('VectorStorageDocumentRequiredError', 'Vector storage requires at least one document for this method');
const VectorStorageDocumentKeyMissingError = StructuredError('VectorStorageDocumentKeyMissingError', 'Vector storage requires each document to have a non-empty key');
const VectorStorageInvalidError = StructuredError('VectorStorageInvalidError', 'Vector storage requires each document to have either embeddings or document property');
const VectorStorageEmbeddingInvalidError = StructuredError('VectorStorageEmbeddingInvalidError', 'Vector storage requires each embeddings property to have a non-empty array of numbers');
const VectorStorageDocumentInvalidError = StructuredError('VectorStorageDocumentInvalidError', 'Vector storage requires each document property to have a non-empty string value');
const VectorStorageResponseError = StructuredError('VectorStorageResponseError')();
export class VectorStorageService {
    #adapter;
    #baseUrl;
    constructor(baseUrl, adapter) {
        this.#adapter = adapter;
        this.#baseUrl = baseUrl;
    }
    async upsert(name, ...documents) {
        if (!name || typeof name !== 'string' || name.trim().length === 0) {
            throw new VectorStorageNameRequiredError();
        }
        if (!documents || documents.length === 0) {
            throw new VectorStorageDocumentRequiredError();
        }
        for (const doc of documents) {
            if (!doc.key || typeof doc.key !== 'string' || doc.key.trim().length === 0) {
                throw new VectorStorageDocumentKeyMissingError();
            }
            if (!('embeddings' in doc) && !('document' in doc)) {
                throw new VectorStorageInvalidError();
            }
            if ('embeddings' in doc && doc.embeddings) {
                if (!Array.isArray(doc.embeddings) || doc.embeddings.length === 0) {
                    throw new VectorStorageEmbeddingInvalidError();
                }
            }
            if ('document' in doc && doc.document) {
                if (typeof doc.document !== 'string' || doc.document.trim().length === 0) {
                    throw new VectorStorageDocumentInvalidError();
                }
            }
        }
        const url = buildUrl(this.#baseUrl, `/vector/2025-03-17/${encodeURIComponent(name)}`);
        const signal = AbortSignal.timeout(30_000);
        // Transform documents to handle TTL: null → 0 for "no expiration"
        const requestDocs = documents.map((doc) => ({
            ...doc,
            // TTL handling: only include if explicitly provided
            // null or 0 = no expiration (send 0 to server), positive = TTL in seconds
            // undefined = not sent, server uses default (30 days)
            ...(doc.ttl !== undefined && { ttl: doc.ttl === null ? 0 : doc.ttl }),
        }));
        const res = await this.#adapter.invoke(url, {
            method: 'PUT',
            body: safeStringify(requestDocs),
            contentType: 'application/json',
            signal,
            telemetry: {
                name: 'agentuity.vector.upsert',
                attributes: {
                    name,
                    count: String(documents.length),
                },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data.map((o, index) => ({
                    key: documents[index].key,
                    id: o.id,
                }));
            }
            throw new VectorStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('PUT', url, res.response);
    }
    async get(name, key) {
        if (!name || typeof name !== 'string' || name.trim().length === 0) {
            throw new VectorStorageNameRequiredError();
        }
        if (!key || typeof key !== 'string' || key.trim().length === 0) {
            throw new VectorStorageKeyRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/vector/2025-03-17/${encodeURIComponent(name)}/${encodeURIComponent(key)}`);
        const signal = AbortSignal.timeout(30_000); // 30s timeout for Neon cold starts
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.vector.get',
                attributes: {
                    name,
                    key,
                },
            },
        });
        if (res.response.status === 404) {
            return { exists: false };
        }
        if (res.ok) {
            if (res.data.success) {
                return {
                    data: res.data.data,
                    exists: true,
                };
            }
            throw new VectorStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('GET', url, res.response);
    }
    async getMany(name, ...keys) {
        if (!name || typeof name !== 'string' || name.trim().length === 0) {
            throw new VectorStorageNameRequiredError();
        }
        if (keys.length === 0) {
            return new Map();
        }
        for (const key of keys) {
            if (!key || typeof key !== 'string' || key.trim().length === 0) {
                throw new VectorStorageKeysError();
            }
        }
        const results = await Promise.all(keys.map((key) => this.get(name, key)));
        const resultMap = new Map();
        results.forEach((result, index) => {
            if (result.exists) {
                resultMap.set(keys[index], result.data);
            }
        });
        return resultMap;
    }
    async search(name, params) {
        if (!name || typeof name !== 'string' || name.trim().length === 0) {
            throw new VectorStorageNameRequiredError();
        }
        if (!params.query || typeof params.query !== 'string' || params.query.trim().length === 0) {
            throw new VectorStorageQueryError();
        }
        if (params.limit !== undefined) {
            if (typeof params.limit !== 'number' || params.limit <= 0) {
                throw new VectorStorageLimitError();
            }
        }
        if (params.similarity !== undefined) {
            if (typeof params.similarity !== 'number' ||
                params.similarity < 0 ||
                params.similarity > 1) {
                throw new VectorStorageSimilarityError();
            }
        }
        if (params.metadata !== undefined) {
            if (typeof params.metadata !== 'object' || params.metadata === null) {
                throw new VectorStorageMetadataError();
            }
        }
        const url = buildUrl(this.#baseUrl, `/vector/2025-03-17/search/${encodeURIComponent(name)}`);
        const signal = AbortSignal.timeout(30_000);
        const attributes = {
            name,
            query: params.query,
        };
        if (params.limit !== undefined) {
            attributes.limit = String(params.limit);
        }
        if (params.similarity !== undefined) {
            attributes.similarity = String(params.similarity);
        }
        const res = await this.#adapter.invoke(url, {
            method: 'POST',
            body: safeStringify(params),
            contentType: 'application/json',
            signal,
            telemetry: {
                name: 'agentuity.vector.search',
                attributes,
            },
        });
        if (res.response.status === 404) {
            return [];
        }
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new VectorStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('POST', url, res.response);
    }
    async delete(name, ...keys) {
        if (!name || typeof name !== 'string' || name.trim().length === 0) {
            throw new VectorStorageNameRequiredError();
        }
        if (keys.length === 0) {
            return 0;
        }
        for (const key of keys) {
            if (!key || typeof key !== 'string' || key.trim().length === 0) {
                throw new VectorStorageKeysError();
            }
        }
        const signal = AbortSignal.timeout(30_000);
        let url;
        let body;
        if (keys.length === 1) {
            url = buildUrl(this.#baseUrl, `/vector/2025-03-17/${encodeURIComponent(name)}/${encodeURIComponent(keys[0])}`);
        }
        else {
            url = buildUrl(this.#baseUrl, `/vector/2025-03-17/${encodeURIComponent(name)}`);
            body = safeStringify({ keys });
        }
        const res = await this.#adapter.invoke(url, {
            method: 'DELETE',
            ...(body && { body, contentType: 'application/json' }),
            signal,
            telemetry: {
                name: 'agentuity.vector.delete',
                attributes: {
                    name,
                    count: String(keys.length),
                },
            },
        });
        // Treat missing vectors as non-fatal for idempotency, like get()/search()
        if (res.response.status === 404) {
            return 0;
        }
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new VectorStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('DELETE', url, res.response);
    }
    async exists(name) {
        if (!name || typeof name !== 'string' || name.trim().length === 0) {
            throw new VectorStorageNameRequiredError();
        }
        const stats = await this.getStats(name);
        return stats.count > 0;
    }
    async getStats(name) {
        if (!name || typeof name !== 'string' || name.trim().length === 0) {
            throw new VectorStorageNameRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/vector/2025-03-17/stats/${encodeURIComponent(name)}`);
        const signal = AbortSignal.timeout(30_000); // 30s timeout for Neon cold starts
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.vector.getStats',
                attributes: {
                    name,
                },
            },
        });
        if (res.ok) {
            return res.data;
        }
        if (res.response.status === 404) {
            return { sum: 0, count: 0 };
        }
        throw await toServiceException('GET', url, res.response);
    }
    async getAllStats(params) {
        const queryParams = new URLSearchParams();
        if (params?.limit !== undefined) {
            queryParams.set('limit', String(params.limit));
        }
        if (params?.offset !== undefined) {
            queryParams.set('offset', String(params.offset));
        }
        const queryString = queryParams.toString();
        const url = buildUrl(this.#baseUrl, `/vector/2025-03-17/stats${queryString ? `?${queryString}` : ''}`);
        const signal = AbortSignal.timeout(30_000); // 30s timeout for Neon cold starts
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.vector.getAllStats',
                attributes: {},
            },
        });
        if (res.ok) {
            return res.data;
        }
        throw await toServiceException('GET', url, res.response);
    }
    async getNamespaces() {
        const url = buildUrl(this.#baseUrl, '/vector/2025-03-17/namespaces');
        const signal = AbortSignal.timeout(30_000); // 30s timeout for Neon cold starts
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.vector.getNamespaces',
                attributes: {},
            },
        });
        if (res.ok) {
            return res.data;
        }
        throw await toServiceException('GET', url, res.response);
    }
    async deleteNamespace(name) {
        if (!name || typeof name !== 'string' || name.trim().length === 0) {
            throw new VectorStorageNameRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/vector/2025-03-17/${encodeURIComponent(name)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'DELETE',
            signal,
            telemetry: {
                name: 'agentuity.vector.deleteNamespace',
                attributes: {
                    name,
                },
            },
        });
        if (res.response.status === 404) {
            return;
        }
        if (res.ok) {
            if (res.data.success) {
                return;
            }
            throw new VectorStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('DELETE', url, res.response);
    }
}
//# sourceMappingURL=vector.js.map