/**
 * Internal Logger for CLI command tracing
 *
 * This logger captures all CLI execution details for debugging purposes.
 * It maintains two files per command execution:
 * 1. session.json - Command metadata, environment, and system info
 * 2. logs.jsonl - JSON Lines format log entries
 *
 * The logger automatically cleans up old logs, keeping only the most recent execution.
 */
import type { Logger } from '@agentuity/core';
/**
 * Internal logger for capturing all CLI command execution details
 */
export declare class InternalLogger implements Logger {
    private cliVersion;
    private cliName;
    private sessionId;
    private sessionDir;
    private sessionFile;
    private logsFile;
    private initialized;
    private disabled;
    constructor(cliVersion: string, cliName: string);
    /**
     * Initialize the internal logger with command metadata
     * @param command - The command being executed
     * @param args - Command line arguments
     * @param userId - Optional user ID (set later via setUserId if not provided)
     * @param projectDir - Optional project directory from --dir flag (defaults to process.cwd())
     */
    init(command: string, args: string[], userId?: string, projectDir?: string): void;
    /**
     * Write a log entry to the logs file
     */
    private writeLog;
    trace(message: unknown, ...args: unknown[]): void;
    debug(message: unknown, ...args: unknown[]): void;
    info(message: unknown, ...args: unknown[]): void;
    warn(message: unknown, ...args: unknown[]): void;
    error(message: unknown, ...args: unknown[]): void;
    fatal(message: unknown, ...args: unknown[]): never;
    child(_opts: Record<string, unknown>): Logger;
    /**
     * Get the session ID for this logger
     */
    getSessionId(): string;
    /**
     * Get the session directory path
     */
    getSessionDir(): string;
    /**
     * Check if the logger is disabled
     */
    isDisabled(): boolean;
    /**
     * Update the session with user ID after authentication
     */
    setUserId(userId: string): void;
    /**
     * Update the session with detected agent name
     */
    setDetectedAgent(agent: string): void;
    /**
     * Disable the internal logger (prevents init and logging)
     */
    disable(): void;
}
/**
 * Create a new internal logger instance
 */
export declare function createInternalLogger(cliVersion: string, cliName: string): InternalLogger;
/**
 * Get the latest log session directory (if any)
 */
export declare function getLatestLogSession(): string | null;
/**
 * Get the logs directory path (exported for external use)
 */
export declare function getLogsDirPath(): string;
//# sourceMappingURL=internal-logger.d.ts.map