/**
 * Map of process names to internal agent short names.
 * The key is the process name (or substring) that appears in the parent process command line.
 * The value is the internal short name used to identify the agent.
 *
 * Process names verified via `agentuity cloud sandbox run --runtime <agent>:latest`:
 * - opencode: binary 'opencode' (from bun install -g opencode-ai)
 * - codex: binary 'codex' (from npm install -g @openai/codex)
 * - cursor: binary 'cursor-agent' (from curl installer)
 * - claude-code: binary 'claude', shows as 'node /usr/local/bin/claude'
 * - copilot: binary 'copilot', shows as 'node /usr/local/bin/copilot' and spawns native binary
 * - gemini: binary 'gemini', shows as 'node /usr/local/bin/gemini'
 * - amp: binary 'amp', shows as 'node --no-warnings /usr/local/bin/amp'
 *
 * IMPORTANT: Order matters! More specific patterns should come before less specific ones.
 * For example, 'opencode' must be checked before 'code' to avoid false matches.
 */
export declare const KNOWN_AGENTS: [string, string][];
export type KnownAgent = (typeof KNOWN_AGENTS)[number][1];
/**
 * Start agent detection immediately (non-blocking).
 * Call this early in CLI startup to begin detection in the background.
 */
export declare function startAgentDetection(): void;
/**
 * Register a callback to be invoked when agent detection completes.
 * If detection has already completed, the callback is invoked immediately.
 * This is non-blocking and does not return a promise.
 *
 * @example
 * ```typescript
 * onAgentDetected((agent) => {
 *   if (agent) {
 *     console.log(`Detected agent: ${agent}`);
 *   }
 * });
 * ```
 */
export declare function onAgentDetected(callback: (agent: string | undefined) => void): void;
/**
 * Get the cached detection result synchronously.
 * Returns undefined if detection hasn't completed yet or no agent was detected.
 * Returns null if detection hasn't started or completed yet.
 *
 * Use this for synchronous access when you don't want to wait for detection.
 */
export declare function getDetectedAgent(): string | undefined | null;
/**
 * Wait for agent detection to complete and ensure all callbacks have been invoked.
 * Call this before CLI exit to ensure the detected agent is written to session logs.
 *
 * This is a no-op if detection hasn't started or has already completed.
 */
export declare function flushAgentDetection(): Promise<void>;
/**
 * Check if the CLI is being executed from a known coding agent.
 * Returns the agent name if detected, undefined otherwise.
 *
 * This function returns immediately if detection has already completed,
 * otherwise it awaits the detection promise started by startAgentDetection().
 *
 * @example
 * ```typescript
 * const agent = await isExecutingFromAgent();
 * if (agent) {
 *   logger.debug(`Running from agent: ${agent}`);
 * }
 * ```
 */
export declare function isExecutingFromAgent(): Promise<string | undefined>;
//# sourceMappingURL=agent-detection.d.ts.map