import { jsx as _jsx } from "react/jsx-runtime";
/**
 * Auth React integration for @agentuity/auth.
 *
 * All React-specific code for auth.
 * Import from '@agentuity/auth/react' for React components and hooks.
 *
 * @module agentuity/react
 */
import { useEffect, createContext, useContext, useState, useMemo } from 'react';
import { createAuthClient as createBetterAuthClient } from 'better-auth/react';
import { organizationClient, apiKeyClient } from 'better-auth/client/plugins';
import { useAuth as useAgentuityReactAuth, useAnalytics } from '@agentuity/react';
/**
 * Get the default client plugins for auth.
 *
 * These mirror the server-side plugins:
 * - organizationClient: Multi-tenancy support
 * - apiKeyClient: Programmatic API key management
 *
 * Note: jwt() and bearer() are server-only plugins.
 */
export function getDefaultClientPlugins() {
    return [organizationClient(), apiKeyClient()];
}
/**
 * Create a pre-configured Auth client.
 *
 * This factory provides sensible defaults for Agentuity projects:
 * - Uses `/api/auth` as the default base path
 * - Automatically uses `window.location.origin` as base URL in browsers
 * - Includes organization and API key plugins by default
 *
 * @example Basic usage (zero config)
 * ```typescript
 * import { createAuthClient } from '@agentuity/auth/react';
 *
 * export const authClient = createAuthClient();
 * export const { signIn, signUp, signOut, useSession, getSession } = authClient;
 * ```
 *
 * @example With custom base path
 * ```typescript
 * export const authClient = createAuthClient({
 *   basePath: '/auth', // If mounted at /auth instead of /api/auth
 * });
 * ```
 *
 * @example With additional plugins
 * ```typescript
 * import { twoFactorClient } from 'better-auth/client/plugins';
 *
 * export const authClient = createAuthClient({
 *   plugins: [twoFactorClient()],
 * });
 * ```
 *
 * @example With custom plugins only (no defaults)
 * ```typescript
 * import { organizationClient } from 'better-auth/client/plugins';
 *
 * export const authClient = createAuthClient({
 *   skipDefaultPlugins: true,
 *   plugins: [organizationClient()],
 * });
 * ```
 */
export function createAuthClient(options) {
    const baseURL = options?.baseURL ?? (typeof window !== 'undefined' ? window.location.origin : '');
    const basePath = options?.basePath ?? '/api/auth';
    const defaultPlugins = options?.skipDefaultPlugins ? [] : getDefaultClientPlugins();
    const userPlugins = options?.plugins ?? [];
    // Merge default plugins with user plugins
    // We pass through the full options to preserve type inference
    // The return type preserves plugin type inference via the generic parameter
    return createBetterAuthClient({
        ...options,
        baseURL,
        basePath,
        plugins: [...defaultPlugins, ...userPlugins],
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
    });
}
const AuthContext = createContext(null);
/**
 * Auth provider component.
 *
 * This component integrates Auth with Agentuity's React context,
 * automatically injecting auth tokens into API calls via useAgent and useWebsocket.
 *
 * Must be a child of AgentuityProvider.
 *
 * @example
 * ```tsx
 * import { AgentuityProvider } from '@agentuity/react';
 * import { createAuthClient, AuthProvider } from '@agentuity/auth/react';
 *
 * const authClient = createAuthClient();
 *
 * <AgentuityProvider>
 *   <AuthProvider authClient={authClient}>
 *     <App />
 *   </AuthProvider>
 * </AgentuityProvider>
 * ```
 */
export function AuthProvider({ children, authClient, refreshInterval = 3600000, tokenEndpoint = '/token', }) {
    const { setAuthHeader, setAuthLoading } = useAgentuityReactAuth();
    const { identify } = useAnalytics();
    const [user, setUser] = useState(null);
    const [session, setSession] = useState(null);
    const [isPending, setIsPending] = useState(true);
    const [error, setError] = useState(null);
    useEffect(() => {
        if (!setAuthHeader || !setAuthLoading)
            return;
        const fetchAuthState = async () => {
            try {
                setAuthLoading(true);
                setIsPending(true);
                setError(null);
                // Use the auth client's getSession method
                const result = await authClient.getSession();
                if (result.data?.user) {
                    const authUser = result.data.user;
                    setUser(authUser);
                    setSession(result.data.session ?? null);
                    // Identify user for analytics
                    identify(authUser.id, {
                        email: authUser.email || '',
                        name: authUser.name || '',
                    });
                    // Get the JWT token for API calls (unless disabled)
                    if (tokenEndpoint !== false) {
                        try {
                            const tokenResult = await authClient.$fetch(tokenEndpoint, { method: 'GET' });
                            const tokenData = tokenResult.data;
                            if (tokenData?.token) {
                                setAuthHeader(`Bearer ${tokenData.token}`);
                            }
                            else {
                                setAuthHeader(null);
                            }
                        }
                        catch {
                            // Token endpoint might not exist, that's okay
                            setAuthHeader(null);
                        }
                    }
                    else {
                        setAuthHeader(null);
                    }
                }
                else {
                    setUser(null);
                    setSession(null);
                    setAuthHeader(null);
                }
            }
            catch (err) {
                console.error('[AuthProvider] Failed to get auth state:', err);
                setError(err instanceof Error ? err : new Error('Failed to get auth state'));
                setUser(null);
                setSession(null);
                setAuthHeader(null);
            }
            finally {
                setAuthLoading(false);
                setIsPending(false);
            }
        };
        fetchAuthState();
        const interval = setInterval(fetchAuthState, refreshInterval);
        return () => clearInterval(interval);
    }, [authClient, refreshInterval, tokenEndpoint, setAuthHeader, setAuthLoading]);
    const contextValue = useMemo(() => ({
        authClient,
        user,
        session,
        isPending,
        error,
        isAuthenticated: !isPending && user !== null,
    }), [authClient, user, session, isPending, error]);
    return _jsx(AuthContext.Provider, { value: contextValue, children: children });
}
/**
 * Hook to access Auth state.
 *
 * This hook provides access to the current user and session.
 * Must be used within an AuthProvider.
 *
 * @example
 * ```tsx
 * import { useAuth } from '@agentuity/auth/react';
 *
 * function Profile() {
 *   const { user, session, isPending, isAuthenticated } = useAuth();
 *
 *   if (isPending) return <div>Loading...</div>;
 *   if (!isAuthenticated) return <div>Not signed in</div>;
 *
 *   return <div>Welcome, {user.name}!</div>;
 * }
 * ```
 */
export function useAuth() {
    const context = useContext(AuthContext);
    if (!context) {
        throw new Error('useAuth must be used within an AuthProvider');
    }
    return context;
}
//# sourceMappingURL=react.js.map