import { createIssue, failure, success, createParseMethods, SCHEMA_KIND } from '../base';
import { optional } from '../utils/optional';
import { nullable } from '../utils/nullable';
const parseMethods = createParseMethods();
/**
 * Schema that coerces values to Date objects using new Date(value).
 * Fails if the result is an invalid date.
 *
 * @example
 * ```typescript
 * const schema = s.coerce.date();
 * schema.parse('2025-01-01'); // Date object
 * schema.parse(1609459200000); // Date from timestamp
 * schema.parse('invalid'); // throws ValidationError
 * ```
 */
export class CoerceDateSchema {
    [SCHEMA_KIND] = 'CoerceDateSchema';
    description;
    '~standard' = {
        version: 1,
        vendor: 'agentuity',
        validate: (value) => {
            // Already a Date
            if (value instanceof Date) {
                if (isNaN(value.getTime())) {
                    return failure([createIssue('Invalid date')]);
                }
                return success(value);
            }
            // Coerce to Date
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            const coerced = new Date(value);
            if (isNaN(coerced.getTime())) {
                return failure([createIssue(`Cannot coerce ${typeof value} to date`)]);
            }
            return success(coerced);
        },
        types: undefined,
    };
    describe(description) {
        this.description = description;
        return this;
    }
    optional() {
        return optional(this);
    }
    nullable() {
        return nullable(this);
    }
    parse = parseMethods.parse;
    safeParse = parseMethods.safeParse;
}
/**
 * Create a schema that coerces values to Date objects.
 * Useful for parsing ISO date strings or timestamps.
 */
export function coerceDate() {
    return new CoerceDateSchema();
}
//# sourceMappingURL=date.js.map