/**
 * Global event bus for Vite-native architecture
 * Replaces the App class event system
 */
import type { Agent, AgentContext } from './agent';
import type { Session, Thread } from './session';
export type AppEventMap<TAppState = Record<string, never>> = {
    'agent.started': [Agent<any, any, any, any, TAppState>, AgentContext<any, any, TAppState>];
    'agent.completed': [Agent<any, any, any, any, TAppState>, AgentContext<any, any, TAppState>];
    'agent.errored': [
        Agent<any, any, any, any, TAppState>,
        AgentContext<any, any, TAppState>,
        Error
    ];
    'session.started': [Session];
    'session.completed': [Session];
    'thread.created': [Thread];
    'thread.destroyed': [Thread];
};
type AppEventCallback<K extends keyof AppEventMap<any>, TAppState = Record<string, never>> = (eventName: K, ...args: AppEventMap<TAppState>[K]) => void | Promise<void>;
/**
 * Register an event listener for application lifecycle events.
 *
 * Available events:
 * - `agent.started` - Fired when an agent begins execution
 * - `agent.completed` - Fired when an agent completes successfully
 * - `agent.errored` - Fired when an agent throws an error
 * - `session.started` - Fired when a new session starts
 * - `session.completed` - Fired when a session completes
 * - `thread.created` - Fired when a thread is created
 * - `thread.destroyed` - Fired when a thread is destroyed
 *
 * @example
 * ```typescript
 * import { addEventListener } from '@agentuity/runtime';
 *
 * addEventListener('agent.started', (eventName, agent, ctx) => {
 *   console.log(`${agent.metadata.name} started for session ${ctx.sessionId}`);
 * });
 * ```
 */
export declare function addEventListener<K extends keyof AppEventMap<any>>(eventName: K, callback: AppEventCallback<K, any>): void;
/**
 * Remove a previously registered event listener.
 */
export declare function removeEventListener<K extends keyof AppEventMap<any>>(eventName: K, callback: AppEventCallback<K, any>): void;
/**
 * Fire a global application event.
 *
 * @example
 * ```typescript
 * import { fireEvent } from '@agentuity/runtime';
 *
 * await fireEvent('session.started', session);
 * await fireEvent('agent.completed', agent, ctx);
 * ```
 */
export declare function fireEvent<K extends keyof AppEventMap<any>>(eventName: K, ...args: AppEventMap<any>[K]): Promise<void>;
/**
 * Clear all event listeners (useful for testing)
 */
export declare function clearAllEventListeners(): void;
export {};
//# sourceMappingURL=_events.d.ts.map