import { type Env as HonoEnv } from 'hono';
import type { cors } from 'hono/cors';
import type { compress } from 'hono/compress';
import type { Logger } from './logger';
import type { Meter, Tracer } from '@opentelemetry/api';
import type { KeyValueStorage, SessionEventProvider, EvalRunEventProvider, StreamStorage, VectorStorage, SandboxService, SessionStartEvent } from '@agentuity/core';
import type { ThreadProvider, SessionProvider, Session, Thread } from './session';
import type WaitUntilHandler from './_waituntil';
import type { Context } from 'hono';
type HonoCorsOptions = NonNullable<Parameters<typeof cors>[0]>;
type HonoCompressOptions = Parameters<typeof compress>[0];
/**
 * Agentuity-specific CORS options for same-origin restriction.
 */
export interface AgentuityCorsSameOriginOptions {
    /**
     * Enable same-origin restriction for CORS.
     *
     * When true, only allows origins from:
     * - AGENTUITY_BASE_URL environment variable
     * - AGENTUITY_CLOUD_DOMAINS environment variable (comma-separated)
     * - AUTH_TRUSTED_DOMAINS environment variable (comma-separated)
     * - The same-origin of the incoming request URL
     * - Any additional origins specified in `allowedOrigins`
     *
     * When false or omitted, the default behavior is to reflect any origin
     * (backwards compatible).
     *
     * @default false
     */
    sameOrigin?: boolean;
    /**
     * Additional origins to allow when `sameOrigin` is true.
     * Can be full URLs (https://example.com) or bare domains (example.com).
     *
     * These are merged with the origins derived from environment variables.
     */
    allowedOrigins?: string[];
}
/**
 * Extended CORS configuration options.
 *
 * Combines Hono's CORS options with Agentuity-specific settings for
 * easy same-origin restriction.
 *
 * @example
 * ```typescript
 * // Simple opt-in to trusted origins only
 * const app = await createApp({
 *   cors: { sameOrigin: true }
 * });
 *
 * // With additional allowed origins
 * const app = await createApp({
 *   cors: {
 *     sameOrigin: true,
 *     allowedOrigins: ['https://admin.myapp.com'],
 *   }
 * });
 * ```
 */
export type CorsConfig = HonoCorsOptions & AgentuityCorsSameOriginOptions;
/**
 * Configuration options for response compression middleware.
 *
 * @example
 * ```typescript
 * const app = await createApp({
 *   compression: {
 *     enabled: true,
 *     threshold: 1024,
 *   }
 * });
 * ```
 */
export interface CompressionConfig {
    /**
     * Enable or disable compression globally.
     * @default true
     */
    enabled?: boolean;
    /**
     * Minimum response body size in bytes before compression is attempted.
     * Responses smaller than this threshold will not be compressed.
     * @default 1024
     */
    threshold?: number;
    /**
     * Optional filter function to skip compression for specific requests.
     * Return false to skip compression for the request.
     *
     * @example
     * ```typescript
     * filter: (c) => !c.req.path.startsWith('/internal')
     * ```
     */
    filter?: (c: Context) => boolean;
    /**
     * Raw options passed through to Hono's compress middleware.
     * These are merged with Agentuity's defaults.
     */
    honoOptions?: HonoCompressOptions;
}
export interface AppConfig<TAppState = Record<string, never>> {
    /**
     * Configure CORS (Cross-Origin Resource Sharing) settings.
     *
     * By default, CORS reflects any origin (backwards compatible).
     * Use `sameOrigin: true` to restrict to trusted origins only.
     *
     * @example
     * ```typescript
     * // Restrict to trusted origins (recommended for production)
     * const app = await createApp({
     *   cors: { sameOrigin: true }
     * });
     *
     * // Add additional allowed origins
     * const app = await createApp({
     *   cors: {
     *     sameOrigin: true,
     *     allowedOrigins: ['https://admin.myapp.com'],
     *   }
     * });
     * ```
     */
    cors?: CorsConfig;
    /**
     * Configure response compression.
     * Set to `false` to disable compression entirely.
     *
     * @example
     * ```typescript
     * const app = await createApp({
     *   compression: {
     *     threshold: 2048,
     *   }
     * });
     *
     * // Or disable compression:
     * const app = await createApp({ compression: false });
     * ```
     */
    compression?: CompressionConfig | false;
    /**
     * Override the default services
     */
    services?: {
        /**
         * if true (default false), will use local services and override any others
         */
        useLocal?: boolean;
        /**
         * the KeyValueStorage to override instead of the default
         */
        keyvalue?: KeyValueStorage;
        /**
         * the StreamStorage to override instead of the default
         */
        stream?: StreamStorage;
        /**
         * the VectorStorage to override instead of the default
         */
        vector?: VectorStorage;
        /**
         * the ThreadProvider to override instead of the default
         */
        thread?: ThreadProvider;
        /**
         * the SessionProvider to override instead of the default
         */
        session?: SessionProvider;
        /**
         * the SessionEventProvider to override instead of the default
         */
        sessionEvent?: SessionEventProvider;
        /**
         * the EvalRunEventProvider to override instead of the default
         */
        evalRunEvent?: EvalRunEventProvider;
    };
    /**
     * Optional setup function called before server starts
     * Returns app state that will be available in all agents and routes
     */
    setup?: () => Promise<TAppState> | TAppState;
    /**
     * Optional shutdown function called when server is stopping
     * Receives the app state returned from setup
     */
    shutdown?: (state: TAppState) => Promise<void> | void;
    /**
     * Optional request timeout in seconds. If not provided, will default
     * to zero which will cause the request to wait indefinitely.
     */
    requestTimeout?: number;
}
export interface Variables<TAppState = Record<string, never>> {
    logger: Logger;
    meter: Meter;
    tracer: Tracer;
    sessionId: string;
    thread: Thread;
    session: Session;
    kv: KeyValueStorage;
    stream: StreamStorage;
    vector: VectorStorage;
    sandbox: SandboxService;
    app: TAppState;
    _webThreadId?: string;
}
export type TriggerType = SessionStartEvent['trigger'];
export interface PrivateVariables {
    waitUntilHandler: WaitUntilHandler;
    routeId?: string;
    agentIds: Set<string>;
    trigger: TriggerType;
    agentRunSpanId?: string;
}
export interface Env<TAppState = Record<string, never>> extends HonoEnv {
    Variables: Variables<TAppState>;
}
/**
 * Get the global app instance (stub for backwards compatibility)
 * Returns null in Vite-native architecture
 */
export declare function getApp(): null;
export { fireEvent } from './_events';
import type { AppEventMap } from './_events';
/**
 * Simple server interface for backwards compatibility
 */
export interface Server {
    /**
     * The server URL (e.g., "http://localhost:3500")
     */
    url: string;
}
export interface AppResult<TAppState = Record<string, never>> {
    /**
     * The application state returned from setup
     */
    state: TAppState;
    /**
     * Shutdown function to call when server stops
     */
    shutdown?: (state: TAppState) => Promise<void> | void;
    /**
     * App configuration (for middleware setup)
     */
    config?: AppConfig<TAppState>;
    /**
     * The router instance (for backwards compatibility)
     */
    router: import('hono').Hono<Env<TAppState>>;
    /**
     * Server information (for backwards compatibility)
     */
    server: Server;
    /**
     * Logger instance (for backwards compatibility)
     */
    logger: Logger;
    /**
     * Add an event listener for app events
     */
    addEventListener<K extends keyof AppEventMap<TAppState>>(eventName: K, callback: (eventName: K, ...args: AppEventMap<TAppState>[K]) => void | Promise<void>): void;
    /**
     * Remove an event listener for app events
     */
    removeEventListener<K extends keyof AppEventMap<TAppState>>(eventName: K, callback: (eventName: K, ...args: AppEventMap<TAppState>[K]) => void | Promise<void>): void;
}
/**
 * Create an Agentuity application with lifecycle management.
 *
 * In Vite-native architecture:
 * - This only handles setup/shutdown lifecycle
 * - Router creation and middleware are handled by the generated entry file
 * - Server is managed by Vite (dev) or Bun.serve (prod)
 *
 * @template TAppState - Type of application state from setup()
 *
 * @example
 * ```typescript
 * // app.ts
 * import { createApp } from '@agentuity/runtime';
 *
 * const app = await createApp({
 *   setup: async () => {
 *     const db = await connectDB();
 *     return { db };
 *   },
 *   shutdown: async (state) => {
 *     await state.db.close();
 *   }
 * });
 *
 * // Access state in agents via ctx.app.db
 * ```
 */
export declare function createApp<TAppState = Record<string, never>>(config?: AppConfig<TAppState>): Promise<AppResult<TAppState>>;
/**
 * Get the global app state
 * Used by generated entry file and middleware
 */
export declare function getAppState<TAppState = any>(): TAppState;
/**
 * Get the global app config
 * Used by generated entry file for middleware setup
 */
export declare function getAppConfig<TAppState = any>(): AppConfig<TAppState> | undefined;
/**
 * Set the global app config (for testing purposes)
 * @internal
 */
export declare function setAppConfig<TAppState = any>(config: AppConfig<TAppState> | undefined): void;
/**
 * Run the global shutdown function
 * Called by generated entry file on cleanup
 */
export declare function runShutdown(): Promise<void>;
//# sourceMappingURL=app.d.ts.map