import type { Schema } from './base';
/**
 * JSON Schema object representation.
 * Subset of JSON Schema Draft 7 specification.
 */
export interface JSONSchema {
    type?: 'string' | 'number' | 'integer' | 'boolean' | 'object' | 'array' | 'null';
    description?: string;
    const?: string | number | boolean;
    enum?: Array<string | number | boolean>;
    properties?: Record<string, JSONSchema>;
    required?: string[];
    items?: JSONSchema;
    anyOf?: JSONSchema[];
    oneOf?: JSONSchema[];
    allOf?: JSONSchema[];
}
/**
 * Convert a schema to JSON Schema format.
 * Supports primitives, objects, arrays, unions, literals, optional, and nullable types.
 *
 * @param schema - The schema to convert
 * @returns JSON Schema object
 *
 * @example
 * ```typescript
 * const userSchema = s.object({
 *   name: s.string().describe('User name'),
 *   age: s.number().describe('User age')
 * });
 *
 * const jsonSchema = s.toJSONSchema(userSchema);
 * // { type: 'object', properties: {...}, required: [...] }
 * ```
 */
export declare function toJSONSchema(schema: Schema<any, any>): JSONSchema;
/**
 * Convert a JSON Schema object to a schema.
 * Supports round-trip conversion with toJSONSchema.
 *
 * @param jsonSchema - The JSON Schema object to convert
 * @returns Schema instance
 *
 * @example
 * ```typescript
 * const jsonSchema = {
 *   type: 'object',
 *   properties: {
 *     name: { type: 'string' },
 *     age: { type: 'number' }
 *   },
 *   required: ['name', 'age']
 * };
 *
 * const schema = s.fromJSONSchema(jsonSchema);
 * const user = schema.parse({ name: 'John', age: 30 });
 * ```
 */
export declare function fromJSONSchema(jsonSchema: JSONSchema): Schema<any, any>;
//# sourceMappingURL=json-schema.d.ts.map