import { EvalHandlerResultSchema, } from '@agentuity/runtime';
import { s } from '@agentuity/schema';
import { generateText } from 'ai';
// Default schemas for preset evals - change these to update all evals
export const DefaultEvalInputSchema = s.object({
    request: s.string(),
    context: s.string().optional(),
});
export const DefaultEvalOutputSchema = s.object({
    response: s.string(),
});
/**
 * Interpolates a prompt template with the provided variables.
 * Replaces {{VARIABLE_NAME}} placeholders with their values.
 *
 * @example
 * ```typescript
 * const prompt = interpolatePrompt(politenessPrompt, {
 *   USER_REQUEST: input.request,
 *   MODEL_RESPONSE: output.response,
 * });
 * ```
 */
export function interpolatePrompt(template, variables) {
    return Object.entries(variables).reduce((prompt, [key, value]) => prompt.replaceAll(`{{${key}}}`, value), template);
}
/**
 * Generates an eval result using LLM with built-in JSON parsing and validation retries.
 *
 * @example
 * ```typescript
 * const result = await generateEvalResult({
 *   model: options.model,
 *   prompt: interpolatePrompt(myPrompt, { ... }),
 * });
 * // result is typed as EvalHandlerResult
 * ```
 */
export async function generateEvalResult(options) {
    const { model, prompt, maxRetries = 3 } = options;
    let lastError;
    for (let attempt = 0; attempt < maxRetries; attempt++) {
        const result = await generateText({ model, prompt });
        try {
            // Extract JSON from response (handles markdown code blocks)
            const jsonMatch = result.text.match(/```(?:json)?\s*([\s\S]*?)```/) || [null, result.text];
            const jsonText = jsonMatch[1]?.trim() || result.text.trim();
            const parsed = JSON.parse(jsonText);
            return EvalHandlerResultSchema.parse(parsed);
        }
        catch (error) {
            lastError = error instanceof Error ? error : new Error(String(error));
            // Don't retry on last attempt
            if (attempt === maxRetries - 1)
                break;
        }
    }
    throw new Error(`Failed to generate valid eval result after ${maxRetries} attempts: ${lastError?.message}`);
}
export function createPresetEval(config) {
    return (overrides) => {
        const { name, description, middleware, ...optionOverrides } = overrides ?? {};
        const currentOptions = { ...config.options, ...optionOverrides };
        return {
            name: name ?? config.name,
            description: description ?? config.description,
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            handler: (async (ctx, input, output) => {
                const evalInput = middleware?.transformInput
                    ? middleware.transformInput(input)
                    : input;
                const evalOutput = middleware?.transformOutput
                    ? middleware.transformOutput(output)
                    : output;
                return config.handler(ctx, evalInput, evalOutput, currentOptions);
                // eslint-disable-next-line @typescript-eslint/no-explicit-any
            }),
            options: currentOptions,
        };
    };
}
//# sourceMappingURL=_utils.js.map