/**
 * Package crypto implements a **FIPS 140-3 compliant KEM-DEM envelope encryption scheme**
 * suitable for multi-gigabyte streams using ECDH P-256 and AES-256-GCM.
 * This design is compatible with the Go implementation and depends only on standard
 * Node.js crypto packages.
 *
 * ──────────────────────────  Design summary  ─────────────────────────────
 *
 *  ⚙  KEM  (Key-Encapsulation Mechanism)
 *      • ECDH P-256 + AES-256-GCM for DEK wrapping
 *      • Output: variable-size encrypted DEK (48-byte DEK + 16-byte GCM tag + ephemeral pubkey)
 *      • Provides forward secrecy for each blob
 *
 *  ⚙  DEM  (Data-Encapsulation Mechanism)
 *      • AES-256-GCM in ~64 KiB framed chunks (65519 bytes max)
 *      • Nonce = 4-byte random prefix ∥ 8-byte little-endian counter
 *      • First frame authenticates header via associated data (prevents tampering)
 *      • Constant ~64 KiB RAM, O(1) header re-wrap for key rotation
 *
 *  ⚙  Fleet key
 *      • Single ECDSA P-256 key-pair per customer
 *      • Public key used directly for ECDH operations
 *      • Private key stored in cloud secret store and fetched at boot
 *
 *  File layout
 *   ┌─────────────────────────────────────────────────────────────────────────┐
 *   │ uint16 wrappedLen │ 125B wrapped DEK │ 12B base nonce │ frames... │
 *   └─────────────────────────────────────────────────────────────────────────┘
 *                               ▲                    ▲
 *                               │                    └─ AES-256-GCM frames
 *                               └─ ECDH + AES-GCM wrapped DEK
 *
 *  Security properties
 *  • Confidentiality & integrity: AES-256-GCM per frame
 *  • Header authentication: first frame includes header as associated data
 *  • Forward-secrecy per object: new ephemeral ECDH key each encryption
 *  • Key rotation: requires re-wrapping only the ~139-byte header
 *  • FIPS 140-3 compliant: uses only approved algorithms
 *
 *  Typical workflow
 *  ────────────────
 *    Publisher:
 *      1) generate DEK, encrypt stream → dst
 *      2) ephemeral ECDH + AES-GCM wrap DEK with fleet public key
 *      3) write header {len, wrapped DEK, nonce} - ~139 bytes total
 *      4) first frame includes header as associated data for authentication
 *
 *    Machine node:
 *      1) read header, unwrap DEK with fleet private key via ECDH
 *      2) stream-decrypt frames on the fly (first frame verifies header)
 *
 * Public API
 * ──────────
 *
 *  encryptFIPSKEMDEMStream(publicKey: KeyObject, src: Readable, dst: Writable): Promise<number>
 *  decryptFIPSKEMDEMStream(privateKey: KeyObject, src: Readable, dst: Writable): Promise<number>
 *
 * Both return the number of plaintext bytes processed and ensure that
 * every error path is authenticated-failure-safe.
 */
import { KeyObject } from 'node:crypto';
import { Readable, Writable } from 'node:stream';
export declare function encryptFIPSKEMDEMStream(pub: KeyObject, src: Readable, dst: Writable): Promise<number>;
export declare function decryptFIPSKEMDEMStream(priv: KeyObject, src: Readable, dst: Writable): Promise<number>;
//# sourceMappingURL=box.d.ts.map