import { clsx } from 'clsx';
import { twMerge } from 'tailwind-merge';
export function cn(...inputs) {
    return twMerge(clsx(inputs));
}
/**
 * Parses the x-agentuity-tokens header string into a Record of model names to token counts.
 * Format: "model1:count1, model2:count2" or "model1:count1"
 * @param header - The x-agentuity-tokens header string
 * @returns Record mapping model names to token counts
 */
export function parseTokensHeader(header) {
    const result = {};
    // Split by comma and trim each entry
    const entries = header.split(',').map((entry) => entry.trim());
    for (const entry of entries) {
        const [model, countStr] = entry.split(':').map((s) => s.trim());
        if (model && countStr) {
            const count = Number.parseInt(countStr, 10);
            if (!Number.isNaN(count)) {
                result[model] = count;
            }
        }
    }
    return result;
}
/**
 * Calculates the total number of tokens from a parsed tokens record.
 * @param tokens - Record mapping model names to token counts
 * @returns Total number of tokens
 */
export function getTotalTokens(tokens) {
    return Object.keys(tokens).reduce((sum, key) => sum + tokens[key], 0);
}
export const getProcessEnv = (key) => {
    // Prioritize import.meta.env for browser/Vite environments
    if (typeof import.meta.env !== 'undefined') {
        return import.meta.env[key];
    }
    if (typeof process !== 'undefined' && process.env) {
        return process.env[key];
    }
    return undefined;
};
export const buildUrl = (base, path, subpath, query) => {
    path = path.startsWith('/') ? path : `/${path}`;
    let url = base.replace(/\/$/, '') + path;
    if (subpath) {
        subpath = subpath.startsWith('/') ? subpath : `/${subpath}`;
        url += subpath;
    }
    if (query) {
        url += `?${query.toString()}`;
    }
    return url;
};
const tryOrigin = () => {
    if (typeof window !== 'undefined') {
        return window.location.origin;
    }
};
export const defaultBaseUrl = getProcessEnv('NEXT_PUBLIC_AGENTUITY_URL') ||
    getProcessEnv('VITE_AGENTUITY_URL') ||
    getProcessEnv('AGENTUITY_URL') ||
    tryOrigin() ||
    'http://localhost:3500';
function generateEmptyValueForSchema(schema) {
    if (typeof schema === 'boolean') {
        return schema ? {} : undefined;
    }
    if (typeof schema !== 'object' || schema === null) {
        return '';
    }
    const s = schema;
    const type = s.type;
    if (Array.isArray(type)) {
        return generateEmptyValueForSchema({ ...s, type: type[0] });
    }
    switch (type) {
        case 'string':
            return '';
        case 'number':
        case 'integer':
            return 0;
        case 'boolean':
            return false;
        case 'null':
            return null;
        case 'array':
            return [];
        case 'object': {
            const result = {};
            if (s.properties) {
                for (const [key, propSchema] of Object.entries(s.properties)) {
                    result[key] = generateEmptyValueForSchema(propSchema);
                }
            }
            return result;
        }
        default:
            if (s.properties) {
                const result = {};
                for (const [key, propSchema] of Object.entries(s.properties)) {
                    result[key] = generateEmptyValueForSchema(propSchema);
                }
                return result;
            }
            return '';
    }
}
export function generateTemplateFromSchema(schema) {
    if (!schema)
        return '{}';
    const template = generateEmptyValueForSchema(schema);
    return JSON.stringify(template, null, 2);
}
export function formatErrorForCopy(error) {
    return [
        error.code ? `[${error.code}] ${error.message}` : error.message,
        error.stack,
        error.cause ? `Cause: ${JSON.stringify(error.cause, null, 2)}` : null,
    ]
        .filter(Boolean)
        .join('\n\n');
}
//# sourceMappingURL=utils.js.map