import { createIssue, failure, success, createParseMethods, SCHEMA_KIND } from '../base';
import { optional } from '../utils/optional';
import { nullable } from '../utils/nullable';
const parseMethods = createParseMethods();
/**
 * Schema that coerces values to numbers using Number(value).
 * Fails if the result is NaN.
 *
 * @example
 * ```typescript
 * const schema = s.coerce.number();
 * schema.parse('123'); // 123
 * schema.parse(true); // 1
 * schema.parse('abc'); // throws ValidationError
 * ```
 */
export class CoerceNumberSchema {
    [SCHEMA_KIND] = 'CoerceNumberSchema';
    description;
    '~standard' = {
        version: 1,
        vendor: 'agentuity',
        validate: (value) => {
            // Already a number
            if (typeof value === 'number') {
                if (Number.isNaN(value)) {
                    return failure([createIssue('Cannot coerce NaN to number')]);
                }
                return success(value);
            }
            // Coerce to number
            const coerced = Number(value);
            if (Number.isNaN(coerced)) {
                return failure([createIssue(`Cannot coerce ${typeof value} to number`)]);
            }
            return success(coerced);
        },
        types: undefined,
    };
    describe(description) {
        this.description = description;
        return this;
    }
    optional() {
        return optional(this);
    }
    nullable() {
        return nullable(this);
    }
    parse = parseMethods.parse;
    safeParse = parseMethods.safeParse;
}
/**
 * Create a schema that coerces values to numbers.
 * Useful for parsing form data or query parameters where numbers come as strings.
 */
export function coerceNumber() {
    return new CoerceNumberSchema();
}
//# sourceMappingURL=number.js.map