import type { InferInput, InferOutput } from '@agentuity/core';
import { type WebSocketRouteRegistry } from '@agentuity/frontend';
/**
 * Extract WebSocket route keys (e.g., '/ws', '/chat')
 */
export type WebSocketRouteKey = keyof WebSocketRouteRegistry;
/**
 * Extract input type for a WebSocket route
 */
export type WebSocketRouteInput<TRoute extends WebSocketRouteKey> = TRoute extends keyof WebSocketRouteRegistry ? WebSocketRouteRegistry[TRoute] extends {
    inputSchema: infer TSchema;
} ? TSchema extends undefined | never ? never : InferInput<TSchema> : never : never;
/**
 * Extract output type for a WebSocket route
 */
export type WebSocketRouteOutput<TRoute extends WebSocketRouteKey> = TRoute extends keyof WebSocketRouteRegistry ? WebSocketRouteRegistry[TRoute] extends {
    outputSchema: infer TSchema;
} ? TSchema extends undefined | never ? void : InferOutput<TSchema> : void : void;
/**
 * Options for WebSocket hooks
 */
export interface WebsocketOptions {
    /**
     * Optional query parameters to append to the websocket URL
     */
    query?: URLSearchParams;
    /**
     * Optional subpath to append to the websocket path
     */
    subpath?: string;
    /**
     * Optional AbortSignal to cancel the websocket connection
     */
    signal?: AbortSignal;
    /**
     * Maximum number of messages to keep in the messages array.
     * When exceeded, oldest messages are removed.
     * @default undefined (no limit)
     */
    maxMessages?: number;
}
/**
 * Type-safe WebSocket hook for connecting to WebSocket routes.
 *
 * Provides automatic type inference for route inputs and outputs based on
 * the WebSocketRouteRegistry generated from your routes.
 *
 * @template TRoute - WebSocket route key from WebSocketRouteRegistry (e.g., '/ws', '/chat')
 *
 * @example Simple WebSocket connection
 * ```typescript
 * const { isConnected, data, send } = useWebsocket('/ws');
 *
 * // Send typed data
 * send({ message: 'Hello' }); // Fully typed based on route schema!
 * ```
 *
 * @example WebSocket with query parameters
 * ```typescript
 * const { isConnected, data, send } = useWebsocket('/chat', {
 *   query: new URLSearchParams({ room: 'general' })
 * });
 * ```
 *
 * @example Access all messages (prevents message loss in rapid succession)
 * ```typescript
 * const { messages, clearMessages } = useWebsocket('/chat');
 *
 * // messages array contains ALL received messages
 * messages.forEach(msg => console.log(msg));
 *
 * // Clear messages when needed
 * clearMessages();
 * ```
 */
export declare function useWebsocket<TRoute extends WebSocketRouteKey>(route: TRoute, options?: WebsocketOptions): {
    isConnected: boolean;
    close: () => void;
    data?: WebSocketRouteOutput<TRoute>;
    messages: WebSocketRouteOutput<TRoute>[];
    clearMessages: () => void;
    error: Error | null;
    isError: boolean;
    reset: () => void;
    send: (data: WebSocketRouteInput<TRoute>) => void;
    readyState: WebSocket['readyState'];
};
//# sourceMappingURL=websocket.d.ts.map