import type { GlobalOptions } from './types';
/**
 * Set global output options (called during CLI initialization)
 */
export declare function setOutputOptions(options: GlobalOptions): void;
/**
 * Get current global output options
 */
export declare function getOutputOptions(): GlobalOptions | null;
/**
 * Check if JSON output mode is enabled
 */
export declare function isJSONMode(options: GlobalOptions): boolean;
/**
 * Check if quiet mode is enabled
 */
export declare function isQuietMode(options: GlobalOptions): boolean;
/**
 * Check if progress indicators should be disabled
 */
export declare function shouldDisableProgress(options: GlobalOptions): boolean;
/**
 * Check if colors should be disabled
 */
export declare function shouldDisableColors(options: GlobalOptions): boolean;
/**
 * Output JSON to stdout (for agent consumption)
 */
export declare function outputJSON(data: unknown): void;
/**
 * Output success message (respects quiet/json mode)
 */
export declare function outputSuccess(message: string, options: GlobalOptions, jsonData?: unknown): void;
/**
 * Output info message (respects quiet/json mode)
 */
export declare function outputInfo(message: string, options: GlobalOptions): void;
/**
 * Output warning message (respects json mode but not quiet - warnings should be seen)
 */
export declare function outputWarning(message: string, options: GlobalOptions): void;
/**
 * Check if interactive prompts should be allowed
 */
export declare function canPrompt(options: GlobalOptions): boolean;
/**
 * Check if validate mode is enabled
 */
export declare function isValidateMode(options: GlobalOptions): boolean;
/**
 * Response metadata
 */
export interface ResponseMetadata {
    timestamp?: string;
    executionTime?: string;
    pagination?: {
        total?: number;
        limit?: number;
        offset?: number;
        hasMore?: boolean;
    };
    warnings?: string[];
}
/**
 * Standard JSON response format
 */
export interface JSONResponse<T = unknown> {
    success: boolean;
    data?: T;
    message?: string;
    metadata?: ResponseMetadata;
    error?: {
        code: string;
        message: string;
        details?: Record<string, unknown>;
    };
}
/**
 * Create a success JSON response
 */
export declare function createSuccessResponse<T>(data?: T, message?: string, metadata?: ResponseMetadata): JSONResponse<T>;
/**
 * Create an error JSON response
 */
export declare function createErrorResponse(code: string, message: string, details?: Record<string, unknown>, metadata?: ResponseMetadata): JSONResponse;
/**
 * Create response metadata with timestamp and optional execution time
 */
export declare function createMetadata(startTime?: number, extra?: Partial<ResponseMetadata>): ResponseMetadata;
/**
 * Batch operation result for a single item
 */
export interface BatchItemResult<T = unknown> {
    item: string;
    success: boolean;
    data?: T;
    error?: {
        code: string;
        message: string;
    };
}
/**
 * Batch operation result
 */
export interface BatchOperationResult<T = unknown> {
    success: boolean;
    totalItems: number;
    succeeded: number;
    failed: number;
    results: BatchItemResult<T>[];
}
/**
 * Create a batch operation result
 */
export declare function createBatchResult<T = unknown>(results: BatchItemResult<T>[]): BatchOperationResult<T>;
/**
 * Output batch operation result
 */
export declare function outputBatchResult<T = unknown>(result: BatchOperationResult<T>, options: GlobalOptions): void;
/**
 * Validation result
 */
export interface ValidationResult {
    valid: boolean;
    command: string;
    errors?: Array<{
        field: string;
        message: string;
    }>;
    warnings?: string[];
}
/**
 * Output validation result
 */
export declare function outputValidation(result: ValidationResult, options: GlobalOptions): void;
//# sourceMappingURL=output.d.ts.map