import { homedir } from 'node:os';
import { join } from 'node:path';
import { YAML } from 'bun';
import { CoderConfigSchema } from '../types';
const CONFIG_DIR = join(homedir(), '.config', 'agentuity');
const DEFAULT_PROFILE = 'production.yaml';
async function getProfilePath() {
    const profileFile = Bun.file(join(CONFIG_DIR, 'profile'));
    if (await profileFile.exists()) {
        const savedPath = (await profileFile.text()).trim();
        const savedFile = Bun.file(savedPath);
        if (await savedFile.exists()) {
            return savedPath;
        }
    }
    return join(CONFIG_DIR, DEFAULT_PROFILE);
}
/**
 * Returns the default config path without resolving the active profile.
 * Use loadCoderConfig() for actual config loading which resolves via getProfilePath().
 */
export function getDefaultConfigPath() {
    return join(CONFIG_DIR, DEFAULT_PROFILE);
}
/**
 * Returns the actual config path that will be used, resolving the active profile.
 */
export async function getConfigPath() {
    return getProfilePath();
}
export async function loadCoderConfig() {
    try {
        const configPath = await getProfilePath();
        const configFile = Bun.file(configPath);
        if (!(await configFile.exists())) {
            return getDefaultConfig();
        }
        const content = await configFile.text();
        const cliConfig = YAML.parse(content);
        const coderConfig = {
            org: cliConfig.preferences?.orgId,
        };
        const result = CoderConfigSchema.safeParse(coderConfig);
        if (!result.success) {
            console.warn(`Warning: Invalid coder config in ${configPath}:`, result.error.message);
            return getDefaultConfig();
        }
        return mergeConfig(getDefaultConfig(), result.data);
    }
    catch (error) {
        console.warn(`Warning: Could not read Agentuity config:`, error);
        return getDefaultConfig();
    }
}
/** Default CLI command patterns to block for security */
const DEFAULT_BLOCKED_COMMANDS = [
    'cloud secrets', // Never expose secrets
    'cloud secret', // Alias
    'cloud apikey', // Don't leak API keys
    'auth token', // Don't leak auth tokens
];
export function getDefaultConfig() {
    return {
        agents: {
            lead: { model: 'anthropic/claude-opus-4-5-20251101' },
            scout: { model: 'anthropic/claude-haiku-4-5-20251001' },
            builder: { model: 'anthropic/claude-opus-4-5-20251101' },
            reviewer: { model: 'anthropic/claude-haiku-4-5-20251001' },
            memory: { model: 'anthropic/claude-haiku-4-5-20251001' },
            expert: { model: 'anthropic/claude-opus-4-5-20251101' },
        },
        disabledMcps: [],
        blockedCommands: DEFAULT_BLOCKED_COMMANDS,
    };
}
export function mergeConfig(base, override) {
    // Deep merge agents: for each agent, merge base and override properties
    const mergedAgents = {};
    const allAgentKeys = new Set([
        ...Object.keys(base.agents ?? {}),
        ...Object.keys(override.agents ?? {}),
    ]);
    for (const key of allAgentKeys) {
        const baseAgent = base.agents?.[key];
        const overrideAgent = override.agents?.[key];
        mergedAgents[key] = {
            ...baseAgent,
            ...overrideAgent,
        };
    }
    return {
        org: override.org ?? base.org,
        agents: mergedAgents,
        disabledMcps: override.disabledMcps ?? base.disabledMcps,
        blockedCommands: override.blockedCommands ?? base.blockedCommands,
    };
}
//# sourceMappingURL=loader.js.map