/**
 * Auth configuration for @agentuity/auth.
 *
 * Provides sensible defaults and wraps BetterAuth with Agentuity-specific helpers.
 *
 * @module agentuity/config
 */
import { type BetterAuthOptions } from 'better-auth';
export type { Organization, OrganizationMember, OrganizationInvitation, OrganizationApiMethods, ApiKey, ApiKeyPluginOptions, ApiKeyApiMethods, JwtApiMethods, DefaultPluginApiMethods, } from './plugins';
export { DEFAULT_API_KEY_OPTIONS } from './plugins';
import type { ApiKeyPluginOptions, DefaultPluginApiMethods } from './plugins';
/**
 * Type for BetterAuth trustedOrigins option.
 * Matches the signature expected by BetterAuthOptions.trustedOrigins.
 */
type TrustedOrigins = string[] | ((request?: Request) => string[] | Promise<string[]>);
/**
 * Base interface for the auth instance used by middleware and route handlers.
 *
 * This interface defines the core authentication APIs that Agentuity middleware
 * relies on. It's designed to be stable and middleware-friendly while the full
 * auth instance provides additional type-safe APIs for your application code.
 *
 * @remarks
 * You typically don't interact with this interface directly. Instead, use
 * `createAuth()` to create an auth instance which implements this interface.
 *
 * @see {@link createAuth} - Create an auth instance
 * @see {@link createSessionMiddleware} - Session-based authentication middleware
 * @see {@link createApiKeyMiddleware} - API key authentication middleware
 */
export interface AuthBase {
    /**
     * HTTP request handler for auth routes.
     *
     * Handles all auth-related endpoints like sign-in, sign-up, sign-out,
     * password reset, OAuth callbacks, and session management.
     *
     * @param request - The incoming HTTP request
     * @returns Response for the auth endpoint
     */
    handler: (request: Request) => Promise<Response>;
    /**
     * Server-side API methods for authentication operations.
     *
     * These methods are used internally by middleware and can also be called
     * directly in your route handlers for custom authentication logic.
     */
    api: {
        /**
         * Get the current session from request headers.
         *
         * @param params - Object containing the request headers
         * @returns The session with user info, or null if not authenticated
         */
        getSession: (params: {
            headers: Headers;
        }) => Promise<{
            user: {
                id: string;
                name?: string | null;
                email: string;
            };
            session: {
                id: string;
                userId: string;
                activeOrganizationId?: string;
            };
        } | null>;
        /**
         * Get full organization details including members.
         *
         * @param params - Object containing the request headers
         * @returns Full organization details, or null if no active org
         */
        getFullOrganization: (params: {
            query?: {
                organizationId?: string;
                organizationSlug?: string;
                membersLimit?: number;
            };
            headers?: Headers;
        }) => Promise<{
            id: string;
            name?: string;
            slug?: string;
            members?: Array<{
                userId: string;
                role: string;
                id?: string;
            }>;
        } | null>;
        /**
         * Verify an API key and get its metadata.
         *
         * @param params - Object containing the API key to verify
         * @returns Validation result with key details if valid
         */
        verifyApiKey: (params: {
            body: {
                key: string;
            };
        }) => Promise<{
            valid: boolean;
            error?: {
                message: string;
                code: string;
            } | null;
            key?: {
                id: string;
                name?: string;
                userId?: string;
                permissions?: Record<string, string[]> | null;
            } | null;
        }>;
    } & DefaultPluginApiMethods;
}
/**
 * Configuration options for auth.
 * Extends BetterAuth options with Agentuity-specific settings.
 *
 * Note: `trustedOrigins` is narrowed to require strict `string[]` (no null/undefined).
 * This ensures type safety for consumers of @agentuity/auth.
 */
export interface AuthOptions extends Omit<BetterAuthOptions, 'trustedOrigins'> {
    /**
     * List of trusted origins for CORS and callback validation.
     * Can be a static array of origin strings, or a function that returns origins.
     *
     * Unlike BetterAuth's type, this requires strict `string[]` with no null/undefined.
     */
    trustedOrigins?: TrustedOrigins;
    /**
     * PostgreSQL connection string.
     * When provided, we create a Bun SQL connection and Drizzle instance internally.
     * This is the simplest path - just provide the connection string.
     *
     * @example
     * ```typescript
     * createAuth({
     *   connectionString: process.env.DATABASE_URL,
     * });
     * ```
     */
    connectionString?: string;
    /**
     * Skip default plugins (organization, jwt, bearer, apiKey).
     * Use this if you want full control over plugins.
     */
    skipDefaultPlugins?: boolean;
    /**
     * API Key plugin configuration.
     * Set to false to disable the API key plugin entirely.
     */
    apiKey?: ApiKeyPluginOptions | false;
}
/**
 * Default plugins included with Agentuity auth.
 *
 * @param apiKeyOptions - API key plugin options, or false to disable
 */
export declare function getDefaultPlugins(apiKeyOptions?: ApiKeyPluginOptions | false): any[];
/**
 * Create an Auth instance.
 *
 * This wraps BetterAuth with sensible defaults for Agentuity projects:
 * - Default basePath: '/api/auth'
 * - Email/password authentication enabled by default
 * - Organization plugin for multi-tenancy
 * - JWT plugin for token-based auth
 * - Bearer plugin for API auth
 * - API Key plugin for programmatic access
 * - Experimental joins enabled by default for better performance
 *
 * @example Option A: Connection string (simplest)
 * ```typescript
 * import { createAuth } from '@agentuity/auth';
 *
 * export const auth = createAuth({
 *   connectionString: process.env.DATABASE_URL,
 * });
 * ```
 *
 * @example Option B: Bring your own Drizzle
 * ```typescript
 * import { drizzle } from 'drizzle-orm/bun-sql';
 * import { drizzleAdapter } from 'better-auth/adapters/drizzle';
 * import * as authSchema from '@agentuity/auth/schema';
 *
 * const schema = { ...authSchema, ...myAppSchema };
 * const db = drizzle(connectionString, { schema });
 *
 * export const auth = createAuth({
 *   database: drizzleAdapter(db, { provider: 'pg', schema: authSchema }),
 * });
 * ```
 *
 * @example Option C: Other adapters (Prisma, MongoDB, etc.)
 * ```typescript
 * import { prismaAdapter } from 'better-auth/adapters/prisma';
 *
 * export const auth = createAuth({
 *   database: prismaAdapter(new PrismaClient()),
 * });
 * ```
 */
export declare function createAuth<T extends AuthOptions>(options: T): AuthBase & import("better-auth").Auth<Omit<T, "plugins" | "connectionString" | "skipDefaultPlugins" | "apiKey"> & {
    trustedOrigins: TrustedOrigins;
    plugins: any[];
    secret?: string | undefined;
    baseURL?: string | undefined;
    database: T["database"] | undefined;
    basePath: NonNullable<T["basePath"]>;
    emailAndPassword: NonNullable<T["emailAndPassword"]> | {
        enabled: true;
    };
    experimental: {
        joins: boolean;
    };
}> & {
    api: import("better-auth").InferAPI<{
        readonly ok: import("better-auth").StrictEndpoint<"/ok", {
            method: "GET";
            metadata: {
                openapi: {
                    description: string;
                    responses: {
                        "200": {
                            description: string;
                            content: {
                                "application/json": {
                                    schema: {
                                        type: "object";
                                        properties: {
                                            ok: {
                                                type: string;
                                                description: string;
                                            };
                                        };
                                        required: string[];
                                    };
                                };
                            };
                        };
                    };
                };
                scope: "server";
            };
        }, {
            ok: boolean;
        }>;
        readonly error: import("better-auth").StrictEndpoint<"/error", {
            method: "GET";
            metadata: {
                openapi: {
                    description: string;
                    responses: {
                        "200": {
                            description: string;
                            content: {
                                "text/html": {
                                    schema: {
                                        type: "string";
                                        description: string;
                                    };
                                };
                            };
                        };
                    };
                };
                scope: "server";
            };
        }, Response>;
        readonly signInSocial: import("better-auth").StrictEndpoint<"/sign-in/social", {
            method: "POST";
            operationId: string;
            body: import("zod").ZodObject<{
                callbackURL: import("zod").ZodOptional<import("zod").ZodString>;
                newUserCallbackURL: import("zod").ZodOptional<import("zod").ZodString>;
                errorCallbackURL: import("zod").ZodOptional<import("zod").ZodString>;
                provider: import("zod").ZodType<"github" | "apple" | "atlassian" | "cognito" | "discord" | "facebook" | "figma" | "microsoft" | "google" | "huggingface" | "slack" | "spotify" | "twitch" | "twitter" | "dropbox" | "kick" | "linear" | "linkedin" | "gitlab" | "tiktok" | "reddit" | "roblox" | "salesforce" | "vk" | "zoom" | "notion" | "kakao" | "naver" | "line" | "paybin" | "paypal" | "polar" | "vercel" | (string & {}), unknown, import("better-auth").$ZodTypeInternals<"github" | "apple" | "atlassian" | "cognito" | "discord" | "facebook" | "figma" | "microsoft" | "google" | "huggingface" | "slack" | "spotify" | "twitch" | "twitter" | "dropbox" | "kick" | "linear" | "linkedin" | "gitlab" | "tiktok" | "reddit" | "roblox" | "salesforce" | "vk" | "zoom" | "notion" | "kakao" | "naver" | "line" | "paybin" | "paypal" | "polar" | "vercel" | (string & {}), unknown>>;
                disableRedirect: import("zod").ZodOptional<import("zod").ZodBoolean>;
                idToken: import("zod").ZodOptional<import("zod").ZodObject<{
                    token: import("zod").ZodString;
                    nonce: import("zod").ZodOptional<import("zod").ZodString>;
                    accessToken: import("zod").ZodOptional<import("zod").ZodString>;
                    refreshToken: import("zod").ZodOptional<import("zod").ZodString>;
                    expiresAt: import("zod").ZodOptional<import("zod").ZodNumber>;
                }, import("better-auth").$strip>>;
                scopes: import("zod").ZodOptional<import("zod").ZodArray<import("zod").ZodString>>;
                requestSignUp: import("zod").ZodOptional<import("zod").ZodBoolean>;
                loginHint: import("zod").ZodOptional<import("zod").ZodString>;
                additionalData: import("zod").ZodOptional<import("zod").ZodRecord<import("zod").ZodString, import("zod").ZodAny>>;
            }, import("better-auth").$strip>;
            metadata: {
                $Infer: {
                    body: import("zod").infer<import("zod").ZodObject<{
                        callbackURL: import("zod").ZodOptional<import("zod").ZodString>;
                        newUserCallbackURL: import("zod").ZodOptional<import("zod").ZodString>;
                        errorCallbackURL: import("zod").ZodOptional<import("zod").ZodString>;
                        provider: import("zod").ZodType<"github" | "apple" | "atlassian" | "cognito" | "discord" | "facebook" | "figma" | "microsoft" | "google" | "huggingface" | "slack" | "spotify" | "twitch" | "twitter" | "dropbox" | "kick" | "linear" | "linkedin" | "gitlab" | "tiktok" | "reddit" | "roblox" | "salesforce" | "vk" | "zoom" | "notion" | "kakao" | "naver" | "line" | "paybin" | "paypal" | "polar" | "vercel" | (string & {}), unknown, import("better-auth").$ZodTypeInternals<"github" | "apple" | "atlassian" | "cognito" | "discord" | "facebook" | "figma" | "microsoft" | "google" | "huggingface" | "slack" | "spotify" | "twitch" | "twitter" | "dropbox" | "kick" | "linear" | "linkedin" | "gitlab" | "tiktok" | "reddit" | "roblox" | "salesforce" | "vk" | "zoom" | "notion" | "kakao" | "naver" | "line" | "paybin" | "paypal" | "polar" | "vercel" | (string & {}), unknown>>;
                        disableRedirect: import("zod").ZodOptional<import("zod").ZodBoolean>;
                        idToken: import("zod").ZodOptional<import("zod").ZodObject<{
                            token: import("zod").ZodString;
                            nonce: import("zod").ZodOptional<import("zod").ZodString>;
                            accessToken: import("zod").ZodOptional<import("zod").ZodString>;
                            refreshToken: import("zod").ZodOptional<import("zod").ZodString>;
                            expiresAt: import("zod").ZodOptional<import("zod").ZodNumber>;
                        }, import("better-auth").$strip>>;
                        scopes: import("zod").ZodOptional<import("zod").ZodArray<import("zod").ZodString>>;
                        requestSignUp: import("zod").ZodOptional<import("zod").ZodBoolean>;
                        loginHint: import("zod").ZodOptional<import("zod").ZodString>;
                        additionalData: import("zod").ZodOptional<import("zod").ZodRecord<import("zod").ZodString, import("zod").ZodAny>>;
                    }, import("better-auth").$strip>>;
                    returned: {
                        redirect: boolean;
                        token?: string | undefined;
                        url?: string | undefined;
                        user?: import("better-auth").UnionToIntersection<import("better-auth").StripEmptyObjects<{
                            id: string;
                            createdAt: Date;
                            updatedAt: Date;
                            email: string;
                            emailVerified: boolean;
                            name: string;
                            image?: string | null | undefined;
                        } & (Omit<T, "plugins" | "connectionString" | "skipDefaultPlugins" | "apiKey"> & {
                            trustedOrigins: TrustedOrigins;
                            plugins: any[];
                            secret?: string | undefined;
                            baseURL?: string | undefined;
                            database: T["database"] | undefined;
                            basePath: NonNullable<T["basePath"]>;
                            emailAndPassword: NonNullable<T["emailAndPassword"]> | {
                                enabled: true;
                            };
                            experimental: {
                                joins: boolean;
                            };
                        } extends infer T_1 ? T_1 extends Omit<T, "plugins" | "connectionString" | "skipDefaultPlugins" | "apiKey"> & {
                            trustedOrigins: TrustedOrigins;
                            plugins: any[];
                            secret?: string | undefined;
                            baseURL?: string | undefined;
                            database: T["database"] | undefined;
                            basePath: NonNullable<T["basePath"]>;
                            emailAndPassword: NonNullable<T["emailAndPassword"]> | {
                                enabled: true;
                            };
                            experimental: {
                                joins: boolean;
                            };
                        } ? T_1 extends BetterAuthOptions ? import("better-auth").AdditionalUserFieldsOutput<T_1> : T_1 extends import("better-auth").Auth ? import("better-auth").AdditionalUserFieldsOutput<T_1["options"]> : {} : never : never)>> | undefined;
                    };
                };
                openapi: {
                    description: string;
                    operationId: string;
                    responses: {
                        "200": {
                            description: string;
                            content: {
                                "application/json": {
                                    schema: {
                                        type: "object";
                                        description: string;
                                        properties: {
                                            token: {
                                                type: string;
                                            };
                                            user: {
                                                type: string;
                                                $ref: string;
                                            };
                                            url: {
                                                type: string;
                                            };
                                            redirect: {
                                                type: string;
                                                enum: boolean[];
                                            };
                                        };
                                        required: string[];
                                    };
                                };
                            };
                        };
                    };
                };
            };
        }, {
            redirect: boolean;
            url: string;
        } | {
            redirect: boolean;
            token: string;
            url: undefined;
            user: import("better-auth").UnionToIntersection<import("better-auth").StripEmptyObjects<{
                id: string;
                createdAt: Date;
                updatedAt: Date;
                email: string;
                emailVerified: boolean;
                name: string;
                image?: string | null | undefined;
            } & (Omit<T, "plugins" | "connectionString" | "skipDefaultPlugins" | "apiKey"> & {
                trustedOrigins: TrustedOrigins;
                plugins: any[];
                secret?: string | undefined;
                baseURL?: string | undefined;
                database: T["database"] | undefined;
                basePath: NonNullable<T["basePath"]>;
                emailAndPassword: NonNullable<T["emailAndPassword"]> | {
                    enabled: true;
                };
                experimental: {
                    joins: boolean;
                };
            } extends infer T_2 ? T_2 extends Omit<T, "plugins" | "connectionString" | "skipDefaultPlugins" | "apiKey"> & {
                trustedOrigins: TrustedOrigins;
                plugins: any[];
                secret?: string | undefined;
                baseURL?: string | undefined;
                database: T["database"] | undefined;
                basePath: NonNullable<T["basePath"]>;
                emailAndPassword: NonNullable<T["emailAndPassword"]> | {
                    enabled: true;
                };
                experimental: {
                    joins: boolean;
                };
            } ? T_2 extends BetterAuthOptions ? import("better-auth").AdditionalUserFieldsOutput<T_2> : T_2 extends import("better-auth").Auth ? import("better-auth").AdditionalUserFieldsOutput<T_2["options"]> : {} : never : never)>>;
        }>;
        readonly callbackOAuth: import("better-auth").StrictEndpoint<"/callback/:id", {
            method: ("GET" | "POST")[];
            operationId: string;
            body: import("zod").ZodOptional<import("zod").ZodObject<{
                code: import("zod").ZodOptional<import("zod").ZodString>;
                error: import("zod").ZodOptional<import("zod").ZodString>;
                device_id: import("zod").ZodOptional<import("zod").ZodString>;
                error_description: import("zod").ZodOptional<import("zod").ZodString>;
                state: import("zod").ZodOptional<import("zod").ZodString>;
                user: import("zod").ZodOptional<import("zod").ZodString>;
            }, import("better-auth").$strip>>;
            query: import("zod").ZodOptional<import("zod").ZodObject<{
                code: import("zod").ZodOptional<import("zod").ZodString>;
                error: import("zod").ZodOptional<import("zod").ZodString>;
                device_id: import("zod").ZodOptional<import("zod").ZodString>;
                error_description: import("zod").ZodOptional<import("zod").ZodString>;
                state: import("zod").ZodOptional<import("zod").ZodString>;
                user: import("zod").ZodOptional<import("zod").ZodString>;
            }, import("better-auth").$strip>>;
            metadata: {
                allowedMediaTypes: string[];
                scope: "server";
            };
        }, void>;
        readonly getSession: import("better-auth").StrictEndpoint<"/get-session", {
            method: "GET";
            operationId: string;
            query: import("zod").ZodOptional<import("zod").ZodObject<{
                disableCookieCache: import("zod").ZodOptional<import("zod").ZodCoercedBoolean<unknown>>;
                disableRefresh: import("zod").ZodOptional<import("zod").ZodCoercedBoolean<unknown>>;
            }, import("better-auth").$strip>>;
            requireHeaders: true;
            metadata: {
                openapi: {
                    operationId: string;
                    description: string;
                    responses: {
                        "200": {
                            description: string;
                            content: {
                                "application/json": {
                                    schema: {
                                        type: "object";
                                        nullable: boolean;
                                        properties: {
                                            session: {
                                                $ref: string;
                                            };
                                            user: {
                                                $ref: string;
                                            };
                                        };
                                        required: string[];
                                    };
                                };
                            };
                        };
                    };
                };
            };
        }, {
            session: import("better-auth").UnionToIntersection<import("better-auth").StripEmptyObjects<{
                id: string;
                createdAt: Date;
                updatedAt: Date;
                userId: string;
                expiresAt: Date;
                token: string;
                ipAddress?: string | null | undefined;
                userAgent?: string | null | undefined;
            } & (Omit<T, "plugins" | "connectionString" | "skipDefaultPlugins" | "apiKey"> & {
                trustedOrigins: TrustedOrigins;
                plugins: any[];
                secret?: string | undefined;
                baseURL?: string | undefined;
                database: T["database"] | undefined;
                basePath: NonNullable<T["basePath"]>;
                emailAndPassword: NonNullable<T["emailAndPassword"]> | {
                    enabled: true;
                };
                experimental: {
                    joins: boolean;
                };
            } extends infer T_3 ? T_3 extends Omit<T, "plugins" | "connectionString" | "skipDefaultPlugins" | "apiKey"> & {
                trustedOrigins: TrustedOrigins;
                plugins: any[];
                secret?: string | undefined;
                baseURL?: string | undefined;
                database: T["database"] | undefined;
                basePath: NonNullable<T["basePath"]>;
                emailAndPassword: NonNullable<T["emailAndPassword"]> | {
                    enabled: true;
                };
                experimental: {
                    joins: boolean;
                };
            } ? T_3 extends BetterAuthOptions ? import("better-auth").AdditionalSessionFieldsOutput<T_3> : T_3 extends import("better-auth").Auth ? import("better-auth").AdditionalSessionFieldsOutput<T_3["options"]> : {} : never : never)>>;
            user: import("better-auth").UnionToIntersection<import("better-auth").StripEmptyObjects<{
                id: string;
                createdAt: Date;
                updatedAt: Date;
                email: string;
                emailVerified: boolean;
                name: string;
                image?: string | null | undefined;
            } & (Omit<T, "plugins" | "connectionString" | "skipDefaultPlugins" | "apiKey"> & {
                trustedOrigins: TrustedOrigins;
                plugins: any[];
                secret?: string | undefined;
                baseURL?: string | undefined;
                database: T["database"] | undefined;
                basePath: NonNullable<T["basePath"]>;
                emailAndPassword: NonNullable<T["emailAndPassword"]> | {
                    enabled: true;
                };
                experimental: {
                    joins: boolean;
                };
            } extends infer T_4 ? T_4 extends Omit<T, "plugins" | "connectionString" | "skipDefaultPlugins" | "apiKey"> & {
                trustedOrigins: TrustedOrigins;
                plugins: any[];
                secret?: string | undefined;
                baseURL?: string | undefined;
                database: T["database"] | undefined;
                basePath: NonNullable<T["basePath"]>;
                emailAndPassword: NonNullable<T["emailAndPassword"]> | {
                    enabled: true;
                };
                experimental: {
                    joins: boolean;
                };
            } ? T_4 extends BetterAuthOptions ? import("better-auth").AdditionalUserFieldsOutput<T_4> : T_4 extends import("better-auth").Auth ? import("better-auth").AdditionalUserFieldsOutput<T_4["options"]> : {} : never : never)>>;
        } | null>;
        readonly signOut: import("better-auth").StrictEndpoint<"/sign-out", {
            method: "POST";
            operationId: string;
            requireHeaders: true;
            metadata: {
                openapi: {
                    operationId: string;
                    description: string;
                    responses: {
                        "200": {
                            description: string;
                            content: {
                                "application/json": {
                                    schema: {
                                        type: "object";
                                        properties: {
                                            success: {
                                                type: string;
                                            };
                                        };
                                    };
                                };
                            };
                        };
                    };
                };
            };
        }, {
            success: boolean;
        }>;
        readonly signUpEmail: import("better-auth").StrictEndpoint<"/sign-up/email", {
            method: "POST";
            operationId: string;
            use: ((inputContext: import("better-auth").MiddlewareInputContext<import("better-auth").MiddlewareOptions>) => Promise<void>)[];
            body: import("zod").ZodIntersection<import("zod").ZodObject<{
                name: import("zod").ZodString;
                email: import("zod").ZodEmail;
                password: import("zod").ZodString;
                image: import("zod").ZodOptional<import("zod").ZodString>;
                callbackURL: import("zod").ZodOptional<import("zod").ZodString>;
                rememberMe: import("zod").ZodOptional<import("zod").ZodBoolean>;
            }, import("better-auth").$strip>, import("zod").ZodRecord<import("zod").ZodString, import("zod").ZodAny>>;
            metadata: {
                allowedMediaTypes: string[];
                $Infer: {
                    body: {
                        name: string;
                        email: string;
                        password: string;
                        image?: string | undefined;
                        callbackURL?: string | undefined;
                        rememberMe?: boolean | undefined;
                    } & import("better-auth/db").InferFieldsFromPlugins<Omit<T, "plugins" | "connectionString" | "skipDefaultPlugins" | "apiKey"> & {
                        trustedOrigins: TrustedOrigins;
                        plugins: any[];
                        secret?: string | undefined;
                        baseURL?: string | undefined;
                        database: T["database"] | undefined;
                        basePath: NonNullable<T["basePath"]>;
                        emailAndPassword: NonNullable<T["emailAndPassword"]> | {
                            enabled: true;
                        };
                        experimental: {
                            joins: boolean;
                        };
                    }, "user", "input"> & import("better-auth/db").InferFieldsFromOptions<Omit<T, "plugins" | "connectionString" | "skipDefaultPlugins" | "apiKey"> & {
                        trustedOrigins: TrustedOrigins;
                        plugins: any[];
                        secret?: string | undefined;
                        baseURL?: string | undefined;
                        database: T["database"] | undefined;
                        basePath: NonNullable<T["basePath"]>;
                        emailAndPassword: NonNullable<T["emailAndPassword"]> | {
                            enabled: true;
                        };
                        experimental: {
                            joins: boolean;
                        };
                    }, "user", "input">;
                    returned: {
                        token: string | null;
                        user: import("better-auth").UnionToIntersection<import("better-auth").StripEmptyObjects<{
                            id: string;
                            createdAt: Date;
                            updatedAt: Date;
                            email: string;
                            emailVerified: boolean;
                            name: string;
                            image?: string | null | undefined;
                        } & (Omit<T, "plugins" | "connectionString" | "skipDefaultPlugins" | "apiKey"> & {
                            trustedOrigins: TrustedOrigins;
                            plugins: any[];
                            secret?: string | undefined;
                            baseURL?: string | undefined;
                            database: T["database"] | undefined;
                            basePath: NonNullable<T["basePath"]>;
                            emailAndPassword: NonNullable<T["emailAndPassword"]> | {
                                enabled: true;
                            };
                            experimental: {
                                joins: boolean;
                            };
                        } extends infer T_5 ? T_5 extends Omit<T, "plugins" | "connectionString" | "skipDefaultPlugins" | "apiKey"> & {
                            trustedOrigins: TrustedOrigins;
                            plugins: any[];
                            secret?: string | undefined;
                            baseURL?: string | undefined;
                            database: T["database"] | undefined;
                            basePath: NonNullable<T["basePath"]>;
                            emailAndPassword: NonNullable<T["emailAndPassword"]> | {
                                enabled: true;
                            };
                            experimental: {
                                joins: boolean;
                            };
                        } ? T_5 extends BetterAuthOptions ? import("better-auth").AdditionalUserFieldsOutput<T_5> : T_5 extends import("better-auth").Auth ? import("better-auth").AdditionalUserFieldsOutput<T_5["options"]> : {} : never : never)>>;
                    };
                };
                openapi: {
                    operationId: string;
                    description: string;
                    requestBody: {
                        content: {
                            "application/json": {
                                schema: {
                                    type: "object";
                                    properties: {
                                        name: {
                                            type: string;
                                            description: string;
                                        };
                                        email: {
                                            type: string;
                                            description: string;
                                        };
                                        password: {
                                            type: string;
                                            description: string;
                                        };
                                        image: {
                                            type: string;
                                            description: string;
                                        };
                                        callbackURL: {
                                            type: string;
                                            description: string;
                                        };
                                        rememberMe: {
                                            type: string;
                                            description: string;
                                        };
                                    };
                                    required: string[];
                                };
                            };
                        };
                    };
                    responses: {
                        "200": {
                            description: string;
                            content: {
                                "application/json": {
                                    schema: {
                                        type: "object";
                                        properties: {
                                            token: {
                                                type: string;
                                                nullable: boolean;
                                                description: string;
                                            };
                                            user: {
                                                type: string;
                                                properties: {
                                                    id: {
                                                        type: string;
                                                        description: string;
                                                    };
                                                    email: {
                                                        type: string;
                                                        format: string;
                                                        description: string;
                                                    };
                                                    name: {
                                                        type: string;
                                                        description: string;
                                                    };
                                                    image: {
                                                        type: string;
                                                        format: string;
                                                        nullable: boolean;
                                                        description: string;
                                                    };
                                                    emailVerified: {
                                                        type: string;
                                                        description: string;
                                                    };
                                                    createdAt: {
                                                        type: string;
                                                        format: string;
                                                        description: string;
                                                    };
                                                    updatedAt: {
                                                        type: string;
                                                        format: string;
                                                        description: string;
                                                    };
                                                };
                                                required: string[];
                                            };
                                        };
                                        required: string[];
                                    };
                                };
                            };
                        };
                        "422": {
                            description: string;
                            content: {
                                "application/json": {
                                    schema: {
                                        type: "object";
                                        properties: {
                                            message: {
                                                type: string;
                                            };
                                        };
                                    };
                                };
                            };
                        };
                    };
                };
            };
        }, {
            token: null;
            user: import("better-auth").UnionToIntersection<import("better-auth").StripEmptyObjects<{
                id: string;
                createdAt: Date;
                updatedAt: Date;
                email: string;
                emailVerified: boolean;
                name: string;
                image?: string | null | undefined;
            } & (Omit<T, "plugins" | "connectionString" | "skipDefaultPlugins" | "apiKey"> & {
                trustedOrigins: TrustedOrigins;
                plugins: any[];
                secret?: string | undefined;
                baseURL?: string | undefined;
                database: T["database"] | undefined;
                basePath: NonNullable<T["basePath"]>;
                emailAndPassword: NonNullable<T["emailAndPassword"]> | {
                    enabled: true;
                };
                experimental: {
                    joins: boolean;
                };
            } extends infer T_6 ? T_6 extends Omit<T, "plugins" | "connectionString" | "skipDefaultPlugins" | "apiKey"> & {
                trustedOrigins: TrustedOrigins;
                plugins: any[];
                secret?: string | undefined;
                baseURL?: string | undefined;
                database: T["database"] | undefined;
                basePath: NonNullable<T["basePath"]>;
                emailAndPassword: NonNullable<T["emailAndPassword"]> | {
                    enabled: true;
                };
                experimental: {
                    joins: boolean;
                };
            } ? T_6 extends BetterAuthOptions ? import("better-auth").AdditionalUserFieldsOutput<T_6> : T_6 extends import("better-auth").Auth ? import("better-auth").AdditionalUserFieldsOutput<T_6["options"]> : {} : never : never)>>;
        } | {
            token: string;
            user: import("better-auth").UnionToIntersection<import("better-auth").StripEmptyObjects<{
                id: string;
                createdAt: Date;
                updatedAt: Date;
                email: string;
                emailVerified: boolean;
                name: string;
                image?: string | null | undefined;
            } & (Omit<T, "plugins" | "connectionString" | "skipDefaultPlugins" | "apiKey"> & {
                trustedOrigins: TrustedOrigins;
                plugins: any[];
                secret?: string | undefined;
                baseURL?: string | undefined;
                database: T["database"] | undefined;
                basePath: NonNullable<T["basePath"]>;
                emailAndPassword: NonNullable<T["emailAndPassword"]> | {
                    enabled: true;
                };
                experimental: {
                    joins: boolean;
                };
            } extends infer T_7 ? T_7 extends Omit<T, "plugins" | "connectionString" | "skipDefaultPlugins" | "apiKey"> & {
                trustedOrigins: TrustedOrigins;
                plugins: any[];
                secret?: string | undefined;
                baseURL?: string | undefined;
                database: T["database"] | undefined;
                basePath: NonNullable<T["basePath"]>;
                emailAndPassword: NonNullable<T["emailAndPassword"]> | {
                    enabled: true;
                };
                experimental: {
                    joins: boolean;
                };
            } ? T_7 extends BetterAuthOptions ? import("better-auth").AdditionalUserFieldsOutput<T_7> : T_7 extends import("better-auth").Auth ? import("better-auth").AdditionalUserFieldsOutput<T_7["options"]> : {} : never : never)>>;
        }>;
        readonly signInEmail: import("better-auth").StrictEndpoint<"/sign-in/email", {
            method: "POST";
            operationId: string;
            use: ((inputContext: import("better-auth").MiddlewareInputContext<import("better-auth").MiddlewareOptions>) => Promise<void>)[];
            body: import("zod").ZodObject<{
                email: import("zod").ZodString;
                password: import("zod").ZodString;
                callbackURL: import("zod").ZodOptional<import("zod").ZodString>;
                rememberMe: import("zod").ZodOptional<import("zod").ZodDefault<import("zod").ZodBoolean>>;
            }, import("better-auth").$strip>;
            metadata: {
                allowedMediaTypes: string[];
                $Infer: {
                    body: {
                        email: string;
                        password: string;
                        callbackURL?: string | undefined;
                        rememberMe?: boolean | undefined;
                    };
                    returned: {
                        redirect: boolean;
                        token: string;
                        url?: string | undefined;
                        user: import("better-auth").UnionToIntersection<import("better-auth").StripEmptyObjects<{
                            id: string;
                            createdAt: Date;
                            updatedAt: Date;
                            email: string;
                            emailVerified: boolean;
                            name: string;
                            image?: string | null | undefined;
                        } & (Omit<T, "plugins" | "connectionString" | "skipDefaultPlugins" | "apiKey"> & {
                            trustedOrigins: TrustedOrigins;
                            plugins: any[];
                            secret?: string | undefined;
                            baseURL?: string | undefined;
                            database: T["database"] | undefined;
                            basePath: NonNullable<T["basePath"]>;
                            emailAndPassword: NonNullable<T["emailAndPassword"]> | {
                                enabled: true;
                            };
                            experimental: {
                                joins: boolean;
                            };
                        } extends infer T_8 ? T_8 extends Omit<T, "plugins" | "connectionString" | "skipDefaultPlugins" | "apiKey"> & {
                            trustedOrigins: TrustedOrigins;
                            plugins: any[];
                            secret?: string | undefined;
                            baseURL?: string | undefined;
                            database: T["database"] | undefined;
                            basePath: NonNullable<T["basePath"]>;
                            emailAndPassword: NonNullable<T["emailAndPassword"]> | {
                                enabled: true;
                            };
                            experimental: {
                                joins: boolean;
                            };
                        } ? T_8 extends BetterAuthOptions ? import("better-auth").AdditionalUserFieldsOutput<T_8> : T_8 extends import("better-auth").Auth ? import("better-auth").AdditionalUserFieldsOutput<T_8["options"]> : {} : never : never)>>;
                    };
                };
                openapi: {
                    operationId: string;
                    description: string;
                    responses: {
                        "200": {
                            description: string;
                            content: {
                                "application/json": {
                                    schema: {
                                        type: "object";
                                        description: string;
                                        properties: {
                                            redirect: {
                                                type: string;
                                                enum: boolean[];
                                            };
                                            token: {
                                                type: string;
                                                description: string;
                                            };
                                            url: {
                                                type: string;
                                                nullable: boolean;
                                            };
                                            user: {
                                                type: string;
                                                $ref: string;
                                            };
                                        };
                                        required: string[];
                                    };
                                };
                            };
                        };
                    };
                };
            };
        }, {
            redirect: boolean;
            token: string;
            url?: string | undefined;
            user: import("better-auth").UnionToIntersection<import("better-auth").StripEmptyObjects<{
                id: string;
                createdAt: Date;
                updatedAt: Date;
                email: string;
                emailVerified: boolean;
                name: string;
                image?: string | null | undefined;
            } & (Omit<T, "plugins" | "connectionString" | "skipDefaultPlugins" | "apiKey"> & {
                trustedOrigins: TrustedOrigins;
                plugins: any[];
                secret?: string | undefined;
                baseURL?: string | undefined;
                database: T["database"] | undefined;
                basePath: NonNullable<T["basePath"]>;
                emailAndPassword: NonNullable<T["emailAndPassword"]> | {
                    enabled: true;
                };
                experimental: {
                    joins: boolean;
                };
            } extends infer T_9 ? T_9 extends Omit<T, "plugins" | "connectionString" | "skipDefaultPlugins" | "apiKey"> & {
                trustedOrigins: TrustedOrigins;
                plugins: any[];
                secret?: string | undefined;
                baseURL?: string | undefined;
                database: T["database"] | undefined;
                basePath: NonNullable<T["basePath"]>;
                emailAndPassword: NonNullable<T["emailAndPassword"]> | {
                    enabled: true;
                };
                experimental: {
                    joins: boolean;
                };
            } ? T_9 extends BetterAuthOptions ? import("better-auth").AdditionalUserFieldsOutput<T_9> : T_9 extends import("better-auth").Auth ? import("better-auth").AdditionalUserFieldsOutput<T_9["options"]> : {} : never : never)>>;
        }>;
        readonly resetPassword: import("better-auth").StrictEndpoint<"/reset-password", {
            method: "POST";
            operationId: string;
            query: import("zod").ZodOptional<import("zod").ZodObject<{
                token: import("zod").ZodOptional<import("zod").ZodString>;
            }, import("better-auth").$strip>>;
            body: import("zod").ZodObject<{
                newPassword: import("zod").ZodString;
                token: import("zod").ZodOptional<import("zod").ZodString>;
            }, import("better-auth").$strip>;
            metadata: {
                openapi: {
                    operationId: string;
                    description: string;
                    responses: {
                        "200": {
                            description: string;
                            content: {
                                "application/json": {
                                    schema: {
                                        type: "object";
                                        properties: {
                                            status: {
                                                type: string;
                                            };
                                        };
                                    };
                                };
                            };
                        };
                    };
                };
            };
        }, {
            status: boolean;
        }>;
        readonly verifyPassword: import("better-auth").StrictEndpoint<"/verify-password", {
            method: "POST";
            body: import("zod").ZodObject<{
                password: import("zod").ZodString;
            }, import("better-auth").$strip>;
            metadata: {
                scope: "server";
                openapi: {
                    operationId: string;
                    description: string;
                    responses: {
                        "200": {
                            description: string;
                            content: {
                                "application/json": {
                                    schema: {
                                        type: "object";
                                        properties: {
                                            status: {
                                                type: string;
                                            };
                                        };
                                    };
                                };
                            };
                        };
                    };
                };
            };
            use: ((inputContext: import("better-auth").MiddlewareInputContext<import("better-auth").MiddlewareOptions>) => Promise<{
                session: {
                    session: Record<string, any> & {
                        id: string;
                        createdAt: Date;
                        updatedAt: Date;
                        userId: string;
                        expiresAt: Date;
                        token: string;
                        ipAddress?: string | null | undefined;
                        userAgent?: string | null | undefined;
                    };
                    user: Record<string, any> & {
                        id: string;
                        createdAt: Date;
                        updatedAt: Date;
                        email: string;
                        emailVerified: boolean;
                        name: string;
                        image?: string | null | undefined;
                    };
                };
            }>)[];
        }, {
            status: boolean;
        }>;
        readonly verifyEmail: import("better-auth").StrictEndpoint<"/verify-email", {
            method: "GET";
            operationId: string;
            query: import("zod").ZodObject<{
                token: import("zod").ZodString;
                callbackURL: import("zod").ZodOptional<import("zod").ZodString>;
            }, import("better-auth").$strip>;
            use: ((inputContext: import("better-auth").MiddlewareInputContext<import("better-auth").MiddlewareOptions>) => Promise<void>)[];
            metadata: {
                openapi: {
                    description: string;
                    parameters: ({
                        name: string;
                        in: "query";
                        description: string;
                        required: true;
                        schema: {
                            type: "string";
                        };
                    } | {
                        name: string;
                        in: "query";
                        description: string;
                        required: false;
                        schema: {
                            type: "string";
                        };
                    })[];
                    responses: {
                        "200": {
                            description: string;
                            content: {
                                "application/json": {
                                    schema: {
                                        type: "object";
                                        properties: {
                                            user: {
                                                type: string;
                                                $ref: string;
                                            };
                                            status: {
                                                type: string;
                                                description: string;
                                            };
                                        };
                                        required: string[];
                                    };
                                };
                            };
                        };
                    };
                };
            };
        }, void | {
            status: boolean;
        }>;
        readonly sendVerificationEmail: import("better-auth").StrictEndpoint<"/send-verification-email", {
            method: "POST";
            operationId: string;
            body: import("zod").ZodObject<{
                email: import("zod").ZodEmail;
                callbackURL: import("zod").ZodOptional<import("zod").ZodString>;
            }, import("better-auth").$strip>;
            metadata: {
                openapi: {
                    operationId: string;
                    description: string;
                    requestBody: {
                        content: {
                            "application/json": {
                                schema: {
                                    type: "object";
                                    properties: {
                                        email: {
                                            type: string;
                                            description: string;
                                            example: string;
                                        };
                                        callbackURL: {
                                            type: string;
                                            description: string;
                                            example: string;
                                            nullable: boolean;
                                        };
                                    };
                                    required: string[];
                                };
                            };
                        };
                    };
                    responses: {
                        "200": {
                            description: string;
                            content: {
                                "application/json": {
                                    schema: {
                                        type: "object";
                                        properties: {
                                            status: {
                                                type: string;
                                                description: string;
                                                example: boolean;
                                            };
                                        };
                                    };
                                };
                            };
                        };
                        "400": {
                            description: string;
                            content: {
                                "application/json": {
                                    schema: {
                                        type: "object";
                                        properties: {
                                            message: {
                                                type: string;
                                                description: string;
                                                example: string;
                                            };
                                        };
                                    };
                                };
                            };
                        };
                    };
                };
            };
        }, {
            status: boolean;
        }>;
        readonly changeEmail: import("better-auth").StrictEndpoint<"/change-email", {
            method: "POST";
            body: import("zod").ZodObject<{
                newEmail: import("zod").ZodEmail;
                callbackURL: import("zod").ZodOptional<import("zod").ZodString>;
            }, import("better-auth").$strip>;
            use: ((inputContext: import("better-auth").MiddlewareInputContext<import("better-auth").MiddlewareOptions>) => Promise<{
                session: {
                    session: Record<string, any> & {
                        id: string;
                        createdAt: Date;
                        updatedAt: Date;
                        userId: string;
                        expiresAt: Date;
                        token: string;
                        ipAddress?: string | null | undefined;
                        userAgent?: string | null | undefined;
                    };
                    user: Record<string, any> & {
                        id: string;
                        createdAt: Date;
                        updatedAt: Date;
                        email: string;
                        emailVerified: boolean;
                        name: string;
                        image?: string | null | undefined;
                    };
                };
            }>)[];
            metadata: {
                openapi: {
                    operationId: string;
                    responses: {
                        "200": {
                            description: string;
                            content: {
                                "application/json": {
                                    schema: {
                                        type: "object";
                                        properties: {
                                            user: {
                                                type: string;
                                                $ref: string;
                                            };
                                            status: {
                                                type: string;
                                                description: string;
                                            };
                                            message: {
                                                type: string;
                                                enum: string[];
                                                description: string;
                                                nullable: boolean;
                                            };
                                        };
                                        required: string[];
                                    };
                                };
                            };
                        };
                        "422": {
                            description: string;
                            content: {
                                "application/json": {
                                    schema: {
                                        type: "object";
                                        properties: {
                                            message: {
                                                type: string;
                                            };
                                        };
                                    };
                                };
                            };
                        };
                    };
                };
            };
        }, {
            status: boolean;
        }>;
        readonly changePassword: import("better-auth").StrictEndpoint<"/change-password", {
            method: "POST";
            operationId: string;
            body: import("zod").ZodObject<{
                newPassword: import("zod").ZodString;
                currentPassword: import("zod").ZodString;
                revokeOtherSessions: import("zod").ZodOptional<import("zod").ZodBoolean>;
            }, import("better-auth").$strip>;
            use: ((inputContext: import("better-auth").MiddlewareInputContext<import("better-auth").MiddlewareOptions>) => Promise<{
                session: {
                    session: Record<string, any> & {
                        id: string;
                        createdAt: Date;
                        updatedAt: Date;
                        userId: string;
                        expiresAt: Date;
                        token: string;
                        ipAddress?: string | null | undefined;
                        userAgent?: string | null | undefined;
                    };
                    user: Record<string, any> & {
                        id: string;
                        createdAt: Date;
                        updatedAt: Date;
                        email: string;
                        emailVerified: boolean;
                        name: string;
                        image?: string | null | undefined;
                    };
                };
            }>)[];
            metadata: {
                openapi: {
                    operationId: string;
                    description: string;
                    responses: {
                        "200": {
                            description: string;
                            content: {
                                "application/json": {
                                    schema: {
                                        type: "object";
                                        properties: {
                                            token: {
                                                type: string;
                                                nullable: boolean;
                                                description: string;
                                            };
                                            user: {
                                                type: string;
                                                properties: {
                                                    id: {
                                                        type: string;
                                                        description: string;
                                                    };
                                                    email: {
                                                        type: string;
                                                        format: string;
                                                        description: string;
                                                    };
                                                    name: {
                                                        type: string;
                                                        description: string;
                                                    };
                                                    image: {
                                                        type: string;
                                                        format: string;
                                                        nullable: boolean;
                                                        description: string;
                                                    };
                                                    emailVerified: {
                                                        type: string;
                                                        description: string;
                                                    };
                                                    createdAt: {
                                                        type: string;
                                                        format: string;
                                                        description: string;
                                                    };
                                                    updatedAt: {
                                                        type: string;
                                                        format: string;
                                                        description: string;
                                                    };
                                                };
                                                required: string[];
                                            };
                                        };
                                        required: string[];
                                    };
                                };
                            };
                        };
                    };
                };
            };
        }, {
            token: string | null;
            user: {
                id: string;
                createdAt: Date;
                updatedAt: Date;
                email: string;
                emailVerified: boolean;
                name: string;
                image?: string | null | undefined;
            } & Record<string, any>;
        }>;
        readonly setPassword: import("better-auth").StrictEndpoint<string, {
            method: "POST";
            body: import("zod").ZodObject<{
                newPassword: import("zod").ZodString;
            }, import("better-auth").$strip>;
            use: ((inputContext: import("better-auth").MiddlewareInputContext<import("better-auth").MiddlewareOptions>) => Promise<{
                session: {
                    session: Record<string, any> & {
                        id: string;
                        createdAt: Date;
                        updatedAt: Date;
                        userId: string;
                        expiresAt: Date;
                        token: string;
                        ipAddress?: string | null | undefined;
                        userAgent?: string | null | undefined;
                    };
                    user: Record<string, any> & {
                        id: string;
                        createdAt: Date;
                        updatedAt: Date;
                        email: string;
                        emailVerified: boolean;
                        name: string;
                        image?: string | null | undefined;
                    };
                };
            }>)[];
        }, {
            status: boolean;
        }>;
        readonly updateUser: import("better-auth").StrictEndpoint<"/update-user", {
            method: "POST";
            operationId: string;
            body: import("zod").ZodRecord<import("zod").ZodString, import("zod").ZodAny>;
            use: ((inputContext: import("better-auth").MiddlewareInputContext<import("better-auth").MiddlewareOptions>) => Promise<{
                session: {
                    session: Record<string, any> & {
                        id: string;
                        createdAt: Date;
                        updatedAt: Date;
                        userId: string;
                        expiresAt: Date;
                        token: string;
                        ipAddress?: string | null | undefined;
                        userAgent?: string | null | undefined;
                    };
                    user: Record<string, any> & {
                        id: string;
                        createdAt: Date;
                        updatedAt: Date;
                        email: string;
                        emailVerified: boolean;
                        name: string;
                        image?: string | null | undefined;
                    };
                };
            }>)[];
            metadata: {
                $Infer: {
                    body: Partial<import("better-auth").AdditionalUserFieldsInput<Omit<T, "plugins" | "connectionString" | "skipDefaultPlugins" | "apiKey"> & {
                        trustedOrigins: TrustedOrigins;
                        plugins: any[];
                        secret?: string | undefined;
                        baseURL?: string | undefined;
                        database: T["database"] | undefined;
                        basePath: NonNullable<T["basePath"]>;
                        emailAndPassword: NonNullable<T["emailAndPassword"]> | {
                            enabled: true;
                        };
                        experimental: {
                            joins: boolean;
                        };
                    }>> & {
                        name?: string | undefined;
                        image?: string | undefined | null;
                    };
                };
                openapi: {
                    operationId: string;
                    description: string;
                    requestBody: {
                        content: {
                            "application/json": {
                                schema: {
                                    type: "object";
                                    properties: {
                                        name: {
                                            type: string;
                                            description: string;
                                        };
                                        image: {
                                            type: string;
                                            description: string;
                                            nullable: boolean;
                                        };
                                    };
                                };
                            };
                        };
                    };
                    responses: {
                        "200": {
                            description: string;
                            content: {
                                "application/json": {
                                    schema: {
                                        type: "object";
                                        properties: {
                                            user: {
                                                type: string;
                                                $ref: string;
                                            };
                                        };
                                    };
                                };
                            };
                        };
                    };
                };
            };
        }, {
            status: boolean;
        }>;
        readonly deleteUser: import("better-auth").StrictEndpoint<"/delete-user", {
            method: "POST";
            use: ((inputContext: import("better-auth").MiddlewareInputContext<import("better-auth").MiddlewareOptions>) => Promise<{
                session: {
                    session: Record<string, any> & {
                        id: string;
                        createdAt: Date;
                        updatedAt: Date;
                        userId: string;
                        expiresAt: Date;
                        token: string;
                        ipAddress?: string | null | undefined;
                        userAgent?: string | null | undefined;
                    };
                    user: Record<string, any> & {
                        id: string;
                        createdAt: Date;
                        updatedAt: Date;
                        email: string;
                        emailVerified: boolean;
                        name: string;
                        image?: string | null | undefined;
                    };
                };
            }>)[];
            body: import("zod").ZodObject<{
                callbackURL: import("zod").ZodOptional<import("zod").ZodString>;
                password: import("zod").ZodOptional<import("zod").ZodString>;
                token: import("zod").ZodOptional<import("zod").ZodString>;
            }, import("better-auth").$strip>;
            metadata: {
                openapi: {
                    operationId: string;
                    description: string;
                    requestBody: {
                        content: {
                            "application/json": {
                                schema: {
                                    type: "object";
                                    properties: {
                                        callbackURL: {
                                            type: string;
                                            description: string;
                                        };
                                        password: {
                                            type: string;
                                            description: string;
                                        };
                                        token: {
                                            type: string;
                                            description: string;
                                        };
                                    };
                                };
                            };
                        };
                    };
                    responses: {
                        "200": {
                            description: string;
                            content: {
                                "application/json": {
                                    schema: {
                                        type: "object";
                                        properties: {
                                            success: {
                                                type: string;
                                                description: string;
                                            };
                                            message: {
                                                type: string;
                                                enum: string[];
                                                description: string;
                                            };
                                        };
                                        required: string[];
                                    };
                                };
                            };
                        };
                    };
                };
            };
        }, {
            success: boolean;
            message: string;
        }>;
        readonly requestPasswordReset: import("better-auth").StrictEndpoint<"/request-password-reset", {
            method: "POST";
            body: import("zod").ZodObject<{
                email: import("zod").ZodEmail;
                redirectTo: import("zod").ZodOptional<import("zod").ZodString>;
            }, import("better-auth").$strip>;
            metadata: {
                openapi: {
                    operationId: string;
                    description: string;
                    responses: {
                        "200": {
                            description: string;
                            content: {
                                "application/json": {
                                    schema: {
                                        type: "object";
                                        properties: {
                                            status: {
                                                type: string;
                                            };
                                            message: {
                                                type: string;
                                            };
                                        };
                                    };
                                };
                            };
                        };
                    };
                };
            };
        }, {
            status: boolean;
            message: string;
        }>;
        readonly requestPasswordResetCallback: import("better-auth").StrictEndpoint<"/reset-password/:token", {
            method: "GET";
            operationId: string;
            query: import("zod").ZodObject<{
                callbackURL: import("zod").ZodString;
            }, import("better-auth").$strip>;
            use: ((inputContext: import("better-auth").MiddlewareInputContext<import("better-auth").MiddlewareOptions>) => Promise<void>)[];
            metadata: {
                openapi: {
                    operationId: string;
                    description: string;
                    parameters: ({
                        name: string;
                        in: "path";
                        required: true;
                        description: string;
                        schema: {
                            type: "string";
                        };
                    } | {
                        name: string;
                        in: "query";
                        required: true;
                        description: string;
                        schema: {
                            type: "string";
                        };
                    })[];
                    responses: {
                        "200": {
                            description: string;
                            content: {
                                "application/json": {
                                    schema: {
                                        type: "object";
                                        properties: {
                                            token: {
                                                type: string;
                                            };
                                        };
                                    };
                                };
                            };
                        };
                    };
                };
            };
        }, never>;
        readonly listSessions: import("better-auth").StrictEndpoint<"/list-sessions", {
            method: "GET";
            operationId: string;
            use: ((inputContext: import("better-auth").MiddlewareInputContext<import("better-auth").MiddlewareOptions>) => Promise<{
                session: {
                    session: Record<string, any> & {
                        id: string;
                        createdAt: Date;
                        updatedAt: Date;
                        userId: string;
                        expiresAt: Date;
                        token: string;
                        ipAddress?: string | null | undefined;
                        userAgent?: string | null | undefined;
                    };
                    user: Record<string, any> & {
                        id: string;
                        createdAt: Date;
                        updatedAt: Date;
                        email: string;
                        emailVerified: boolean;
                        name: string;
                        image?: string | null | undefined;
                    };
                };
            }>)[];
            requireHeaders: true;
            metadata: {
                openapi: {
                    operationId: string;
                    description: string;
                    responses: {
                        "200": {
                            description: string;
                            content: {
                                "application/json": {
                                    schema: {
                                        type: "array";
                                        items: {
                                            $ref: string;
                                        };
                                    };
                                };
                            };
                        };
                    };
                };
            };
        }, import("better-auth").Prettify<import("better-auth").UnionToIntersection<import("better-auth").StripEmptyObjects<{
            id: string;
            createdAt: Date;
            updatedAt: Date;
            userId: string;
            expiresAt: Date;
            token: string;
            ipAddress?: string | null | undefined;
            userAgent?: string | null | undefined;
        } & (Omit<T, "plugins" | "connectionString" | "skipDefaultPlugins" | "apiKey"> & {
            trustedOrigins: TrustedOrigins;
            plugins: any[];
            secret?: string | undefined;
            baseURL?: string | undefined;
            database: T["database"] | undefined;
            basePath: NonNullable<T["basePath"]>;
            emailAndPassword: NonNullable<T["emailAndPassword"]> | {
                enabled: true;
            };
            experimental: {
                joins: boolean;
            };
        } extends infer T_10 ? T_10 extends Omit<T, "plugins" | "connectionString" | "skipDefaultPlugins" | "apiKey"> & {
            trustedOrigins: TrustedOrigins;
            plugins: any[];
            secret?: string | undefined;
            baseURL?: string | undefined;
            database: T["database"] | undefined;
            basePath: NonNullable<T["basePath"]>;
            emailAndPassword: NonNullable<T["emailAndPassword"]> | {
                enabled: true;
            };
            experimental: {
                joins: boolean;
            };
        } ? T_10 extends BetterAuthOptions ? import("better-auth").AdditionalSessionFieldsOutput<T_10> : T_10 extends import("better-auth").Auth ? import("better-auth").AdditionalSessionFieldsOutput<T_10["options"]> : {} : never : never)>>>[]>;
        readonly revokeSession: import("better-auth").StrictEndpoint<"/revoke-session", {
            method: "POST";
            body: import("zod").ZodObject<{
                token: import("zod").ZodString;
            }, import("better-auth").$strip>;
            use: ((inputContext: import("better-auth").MiddlewareInputContext<import("better-auth").MiddlewareOptions>) => Promise<{
                session: {
                    session: Record<string, any> & {
                        id: string;
                        createdAt: Date;
                        updatedAt: Date;
                        userId: string;
                        expiresAt: Date;
                        token: string;
                        ipAddress?: string | null | undefined;
                        userAgent?: string | null | undefined;
                    };
                    user: Record<string, any> & {
                        id: string;
                        createdAt: Date;
                        updatedAt: Date;
                        email: string;
                        emailVerified: boolean;
                        name: string;
                        image?: string | null | undefined;
                    };
                };
            }>)[];
            requireHeaders: true;
            metadata: {
                openapi: {
                    description: string;
                    requestBody: {
                        content: {
                            "application/json": {
                                schema: {
                                    type: "object";
                                    properties: {
                                        token: {
                                            type: string;
                                            description: string;
                                        };
                                    };
                                    required: string[];
                                };
                            };
                        };
                    };
                    responses: {
                        "200": {
                            description: string;
                            content: {
                                "application/json": {
                                    schema: {
                                        type: "object";
                                        properties: {
                                            status: {
                                                type: string;
                                                description: string;
                                            };
                                        };
                                        required: string[];
                                    };
                                };
                            };
                        };
                    };
                };
            };
        }, {
            status: boolean;
        }>;
        readonly revokeSessions: import("better-auth").StrictEndpoint<"/revoke-sessions", {
            method: "POST";
            use: ((inputContext: import("better-auth").MiddlewareInputContext<import("better-auth").MiddlewareOptions>) => Promise<{
                session: {
                    session: Record<string, any> & {
                        id: string;
                        createdAt: Date;
                        updatedAt: Date;
                        userId: string;
                        expiresAt: Date;
                        token: string;
                        ipAddress?: string | null | undefined;
                        userAgent?: string | null | undefined;
                    };
                    user: Record<string, any> & {
                        id: string;
                        createdAt: Date;
                        updatedAt: Date;
                        email: string;
                        emailVerified: boolean;
                        name: string;
                        image?: string | null | undefined;
                    };
                };
            }>)[];
            requireHeaders: true;
            metadata: {
                openapi: {
                    description: string;
                    responses: {
                        "200": {
                            description: string;
                            content: {
                                "application/json": {
                                    schema: {
                                        type: "object";
                                        properties: {
                                            status: {
                                                type: string;
                                                description: string;
                                            };
                                        };
                                        required: string[];
                                    };
                                };
                            };
                        };
                    };
                };
            };
        }, {
            status: boolean;
        }>;
        readonly revokeOtherSessions: import("better-auth").StrictEndpoint<"/revoke-other-sessions", {
            method: "POST";
            requireHeaders: true;
            use: ((inputContext: import("better-auth").MiddlewareInputContext<import("better-auth").MiddlewareOptions>) => Promise<{
                session: {
                    session: Record<string, any> & {
                        id: string;
                        createdAt: Date;
                        updatedAt: Date;
                        userId: string;
                        expiresAt: Date;
                        token: string;
                        ipAddress?: string | null | undefined;
                        userAgent?: string | null | undefined;
                    };
                    user: Record<string, any> & {
                        id: string;
                        createdAt: Date;
                        updatedAt: Date;
                        email: string;
                        emailVerified: boolean;
                        name: string;
                        image?: string | null | undefined;
                    };
                };
            }>)[];
            metadata: {
                openapi: {
                    description: string;
                    responses: {
                        "200": {
                            description: string;
                            content: {
                                "application/json": {
                                    schema: {
                                        type: "object";
                                        properties: {
                                            status: {
                                                type: string;
                                                description: string;
                                            };
                                        };
                                        required: string[];
                                    };
                                };
                            };
                        };
                    };
                };
            };
        }, {
            status: boolean;
        }>;
        readonly linkSocialAccount: import("better-auth").StrictEndpoint<"/link-social", {
            method: "POST";
            requireHeaders: true;
            body: import("zod").ZodObject<{
                callbackURL: import("zod").ZodOptional<import("zod").ZodString>;
                provider: import("zod").ZodType<"github" | "apple" | "atlassian" | "cognito" | "discord" | "facebook" | "figma" | "microsoft" | "google" | "huggingface" | "slack" | "spotify" | "twitch" | "twitter" | "dropbox" | "kick" | "linear" | "linkedin" | "gitlab" | "tiktok" | "reddit" | "roblox" | "salesforce" | "vk" | "zoom" | "notion" | "kakao" | "naver" | "line" | "paybin" | "paypal" | "polar" | "vercel" | (string & {}), unknown, import("better-auth").$ZodTypeInternals<"github" | "apple" | "atlassian" | "cognito" | "discord" | "facebook" | "figma" | "microsoft" | "google" | "huggingface" | "slack" | "spotify" | "twitch" | "twitter" | "dropbox" | "kick" | "linear" | "linkedin" | "gitlab" | "tiktok" | "reddit" | "roblox" | "salesforce" | "vk" | "zoom" | "notion" | "kakao" | "naver" | "line" | "paybin" | "paypal" | "polar" | "vercel" | (string & {}), unknown>>;
                idToken: import("zod").ZodOptional<import("zod").ZodObject<{
                    token: import("zod").ZodString;
                    nonce: import("zod").ZodOptional<import("zod").ZodString>;
                    accessToken: import("zod").ZodOptional<import("zod").ZodString>;
                    refreshToken: import("zod").ZodOptional<import("zod").ZodString>;
                    scopes: import("zod").ZodOptional<import("zod").ZodArray<import("zod").ZodString>>;
                }, import("better-auth").$strip>>;
                requestSignUp: import("zod").ZodOptional<import("zod").ZodBoolean>;
                scopes: import("zod").ZodOptional<import("zod").ZodArray<import("zod").ZodString>>;
                errorCallbackURL: import("zod").ZodOptional<import("zod").ZodString>;
                disableRedirect: import("zod").ZodOptional<import("zod").ZodBoolean>;
                additionalData: import("zod").ZodOptional<import("zod").ZodRecord<import("zod").ZodString, import("zod").ZodAny>>;
            }, import("better-auth").$strip>;
            use: ((inputContext: import("better-auth").MiddlewareInputContext<import("better-auth").MiddlewareOptions>) => Promise<{
                session: {
                    session: Record<string, any> & {
                        id: string;
                        createdAt: Date;
                        updatedAt: Date;
                        userId: string;
                        expiresAt: Date;
                        token: string;
                        ipAddress?: string | null | undefined;
                        userAgent?: string | null | undefined;
                    };
                    user: Record<string, any> & {
                        id: string;
                        createdAt: Date;
                        updatedAt: Date;
                        email: string;
                        emailVerified: boolean;
                        name: string;
                        image?: string | null | undefined;
                    };
                };
            }>)[];
            metadata: {
                openapi: {
                    description: string;
                    operationId: string;
                    responses: {
                        "200": {
                            description: string;
                            content: {
                                "application/json": {
                                    schema: {
                                        type: "object";
                                        properties: {
                                            url: {
                                                type: string;
                                                description: string;
                                            };
                                            redirect: {
                                                type: string;
                                                description: string;
                                            };
                                            status: {
                                                type: string;
                                            };
                                        };
                                        required: string[];
                                    };
                                };
                            };
                        };
                    };
                };
            };
        }, {
            url: string;
            redirect: boolean;
        }>;
        readonly listUserAccounts: import("better-auth").StrictEndpoint<"/list-accounts", {
            method: "GET";
            use: ((inputContext: import("better-auth").MiddlewareInputContext<import("better-auth").MiddlewareOptions>) => Promise<{
                session: {
                    session: Record<string, any> & {
                        id: string;
                        createdAt: Date;
                        updatedAt: Date;
                        userId: string;
                        expiresAt: Date;
                        token: string;
                        ipAddress?: string | null | undefined;
                        userAgent?: string | null | undefined;
                    };
                    user: Record<string, any> & {
                        id: string;
                        createdAt: Date;
                        updatedAt: Date;
                        email: string;
                        emailVerified: boolean;
                        name: string;
                        image?: string | null | undefined;
                    };
                };
            }>)[];
            metadata: {
                openapi: {
                    operationId: string;
                    description: string;
                    responses: {
                        "200": {
                            description: string;
                            content: {
                                "application/json": {
                                    schema: {
                                        type: "array";
                                        items: {
                                            type: string;
                                            properties: {
                                                id: {
                                                    type: string;
                                                };
                                                providerId: {
                                                    type: string;
                                                };
                                                createdAt: {
                                                    type: string;
                                                    format: string;
                                                };
                                                updatedAt: {
                                                    type: string;
                                                    format: string;
                                                };
                                                accountId: {
                                                    type: string;
                                                };
                                                userId: {
                                                    type: string;
                                                };
                                                scopes: {
                                                    type: string;
                                                    items: {
                                                        type: string;
                                                    };
                                                };
                                            };
                                            required: string[];
                                        };
                                    };
                                };
                            };
                        };
                    };
                };
            };
        }, {
            scopes: string[];
            id: string;
            createdAt: Date;
            updatedAt: Date;
            userId: string;
            providerId: string;
            accountId: string;
        }[]>;
        readonly deleteUserCallback: import("better-auth").StrictEndpoint<"/delete-user/callback", {
            method: "GET";
            query: import("zod").ZodObject<{
                token: import("zod").ZodString;
                callbackURL: import("zod").ZodOptional<import("zod").ZodString>;
            }, import("better-auth").$strip>;
            use: ((inputContext: import("better-auth").MiddlewareInputContext<import("better-auth").MiddlewareOptions>) => Promise<void>)[];
            metadata: {
                openapi: {
                    description: string;
                    responses: {
                        "200": {
                            description: string;
                            content: {
                                "application/json": {
                                    schema: {
                                        type: "object";
                                        properties: {
                                            success: {
                                                type: string;
                                                description: string;
                                            };
                                            message: {
                                                type: string;
                                                enum: string[];
                                                description: string;
                                            };
                                        };
                                        required: string[];
                                    };
                                };
                            };
                        };
                    };
                };
            };
        }, {
            success: boolean;
            message: string;
        }>;
        readonly unlinkAccount: import("better-auth").StrictEndpoint<"/unlink-account", {
            method: "POST";
            body: import("zod").ZodObject<{
                providerId: import("zod").ZodString;
                accountId: import("zod").ZodOptional<import("zod").ZodString>;
            }, import("better-auth").$strip>;
            use: ((inputContext: import("better-auth").MiddlewareInputContext<import("better-auth").MiddlewareOptions>) => Promise<{
                session: {
                    session: Record<string, any> & {
                        id: string;
                        createdAt: Date;
                        updatedAt: Date;
                        userId: string;
                        expiresAt: Date;
                        token: string;
                        ipAddress?: string | null | undefined;
                        userAgent?: string | null | undefined;
                    };
                    user: Record<string, any> & {
                        id: string;
                        createdAt: Date;
                        updatedAt: Date;
                        email: string;
                        emailVerified: boolean;
                        name: string;
                        image?: string | null | undefined;
                    };
                };
            }>)[];
            metadata: {
                openapi: {
                    description: string;
                    responses: {
                        "200": {
                            description: string;
                            content: {
                                "application/json": {
                                    schema: {
                                        type: "object";
                                        properties: {
                                            status: {
                                                type: string;
                                            };
                                        };
                                    };
                                };
                            };
                        };
                    };
                };
            };
        }, {
            status: boolean;
        }>;
        readonly refreshToken: import("better-auth").StrictEndpoint<"/refresh-token", {
            method: "POST";
            body: import("zod").ZodObject<{
                providerId: import("zod").ZodString;
                accountId: import("zod").ZodOptional<import("zod").ZodString>;
                userId: import("zod").ZodOptional<import("zod").ZodString>;
            }, import("better-auth").$strip>;
            metadata: {
                openapi: {
                    description: string;
                    responses: {
                        200: {
                            description: string;
                            content: {
                                "application/json": {
                                    schema: {
                                        type: "object";
                                        properties: {
                                            tokenType: {
                                                type: string;
                                            };
                                            idToken: {
                                                type: string;
                                            };
                                            accessToken: {
                                                type: string;
                                            };
                                            refreshToken: {
                                                type: string;
                                            };
                                            accessTokenExpiresAt: {
                                                type: string;
                                                format: string;
                                            };
                                            refreshTokenExpiresAt: {
                                                type: string;
                                                format: string;
                                            };
                                        };
                                    };
                                };
                            };
                        };
                        400: {
                            description: string;
                        };
                    };
                };
            };
        }, {
            accessToken: string | undefined;
            refreshToken: string | undefined;
            accessTokenExpiresAt: Date | undefined;
            refreshTokenExpiresAt: Date | undefined;
            scope: string | null | undefined;
            idToken: string | null | undefined;
            providerId: string;
            accountId: string;
        }>;
        readonly getAccessToken: import("better-auth").StrictEndpoint<"/get-access-token", {
            method: "POST";
            body: import("zod").ZodObject<{
                providerId: import("zod").ZodString;
                accountId: import("zod").ZodOptional<import("zod").ZodString>;
                userId: import("zod").ZodOptional<import("zod").ZodString>;
            }, import("better-auth").$strip>;
            metadata: {
                openapi: {
                    description: string;
                    responses: {
                        200: {
                            description: string;
                            content: {
                                "application/json": {
                                    schema: {
                                        type: "object";
                                        properties: {
                                            tokenType: {
                                                type: string;
                                            };
                                            idToken: {
                                                type: string;
                                            };
                                            accessToken: {
                                                type: string;
                                            };
                                            accessTokenExpiresAt: {
                                                type: string;
                                                format: string;
                                            };
                                        };
                                    };
                                };
                            };
                        };
                        400: {
                            description: string;
                        };
                    };
                };
            };
        }, {
            accessToken: string;
            accessTokenExpiresAt: Date | undefined;
            scopes: string[];
            idToken: string | undefined;
        }>;
        readonly accountInfo: import("better-auth").StrictEndpoint<"/account-info", {
            method: "GET";
            use: ((inputContext: import("better-auth").MiddlewareInputContext<import("better-auth").MiddlewareOptions>) => Promise<{
                session: {
                    session: Record<string, any> & {
                        id: string;
                        createdAt: Date;
                        updatedAt: Date;
                        userId: string;
                        expiresAt: Date;
                        token: string;
                        ipAddress?: string | null | undefined;
                        userAgent?: string | null | undefined;
                    };
                    user: Record<string, any> & {
                        id: string;
                        createdAt: Date;
                        updatedAt: Date;
                        email: string;
                        emailVerified: boolean;
                        name: string;
                        image?: string | null | undefined;
                    };
                };
            }>)[];
            metadata: {
                openapi: {
                    description: string;
                    responses: {
                        "200": {
                            description: string;
                            content: {
                                "application/json": {
                                    schema: {
                                        type: "object";
                                        properties: {
                                            user: {
                                                type: string;
                                                properties: {
                                                    id: {
                                                        type: string;
                                                    };
                                                    name: {
                                                        type: string;
                                                    };
                                                    email: {
                                                        type: string;
                                                    };
                                                    image: {
                                                        type: string;
                                                    };
                                                    emailVerified: {
                                                        type: string;
                                                    };
                                                };
                                                required: string[];
                                            };
                                            data: {
                                                type: string;
                                                properties: {};
                                                additionalProperties: boolean;
                                            };
                                        };
                                        required: string[];
                                        additionalProperties: boolean;
                                    };
                                };
                            };
                        };
                    };
                };
            };
            query: import("zod").ZodOptional<import("zod").ZodObject<{
                accountId: import("zod").ZodOptional<import("zod").ZodString>;
            }, import("better-auth").$strip>>;
        }, {
            user: import("better-auth").OAuth2UserInfo;
            data: Record<string, any>;
        } | null>;
    } & import("better-auth").UnionToIntersection<(Omit<T, "plugins" | "connectionString" | "skipDefaultPlugins" | "apiKey"> & {
        trustedOrigins: TrustedOrigins;
        plugins: any[];
        secret?: string | undefined;
        baseURL?: string | undefined;
        database: T["database"] | undefined;
        basePath: NonNullable<T["basePath"]>;
        emailAndPassword: NonNullable<T["emailAndPassword"]> | {
            enabled: true;
        };
        experimental: {
            joins: boolean;
        };
    })["plugins"] extends (infer T_11)[] ? T_11 extends import("better-auth").BetterAuthPlugin ? T_11 extends {
        endpoints: infer E;
    } ? E : {} : {} : {}>> & DefaultPluginApiMethods;
};
/**
 * Type helper for the auth instance with default plugin methods.
 * Inferred from createAuth to stay in sync with BetterAuth.
 */
export type AuthInstance = ReturnType<typeof createAuth>;
//# sourceMappingURL=config.d.ts.map