import { Hono, type Schema, type Env as HonoEnv } from 'hono';
import type { Env } from './app';
export type { Env };
export type { HonoEnv };
export type { WebSocketConnection } from './handlers/websocket';
declare module 'hono' {
    interface Context {
        /**
         * Schedule a background task that runs after the response is sent.
         * Works the same as `ctx.waitUntil()` in agent handlers.
         *
         * @example
         * ```typescript
         * router.post('/data', async (c) => {
         *   c.waitUntil(async () => {
         *     await sendAnalytics(c.req.url);
         *   });
         *   return c.json({ success: true });
         * });
         * ```
         */
        waitUntil(callback: Promise<void> | (() => void | Promise<void>)): void;
    }
    interface Hono {
        /**
         * @deprecated Use the `websocket` middleware instead:
         * ```typescript
         * import { websocket } from '@agentuity/runtime';
         * router.get('/ws', websocket((c, ws) => { ... }));
         * ```
         */
        websocket(path: string, ...args: any[]): this;
        /**
         * @deprecated Use the `sse` middleware instead:
         * ```typescript
         * import { sse } from '@agentuity/runtime';
         * router.get('/events', sse((c, stream) => { ... }));
         * ```
         */
        sse(path: string, ...args: any[]): this;
        /**
         * @deprecated Use the `stream` middleware instead:
         * ```typescript
         * import { stream } from '@agentuity/runtime';
         * router.post('/data', stream((c) => new ReadableStream({ ... })));
         * ```
         */
        stream(path: string, ...args: any[]): this;
        /**
         * @deprecated Use the `cron` middleware instead:
         * ```typescript
         * import { cron } from '@agentuity/runtime';
         * router.post('/job', cron('0 0 * * *', (c) => { ... }));
         * ```
         */
        cron(schedule: string, ...args: any[]): this;
    }
}
/**
 * Creates a Hono router with extended methods for Agentuity-specific routing patterns.
 *
 * Standard HTTP methods (get, post, put, delete, patch) are available, plus middleware
 * functions for specialized protocols:
 *
 * - **websocket()** - WebSocket connections (import { websocket } from '@agentuity/runtime')
 * - **sse()** - Server-Sent Events (import { sse } from '@agentuity/runtime')
 * - **stream()** - Streaming responses (import { stream } from '@agentuity/runtime')
 * - **cron()** - Scheduled tasks (import { cron } from '@agentuity/runtime')
 *
 * @template E - Environment type (Hono Env)
 * @template S - Schema type for route definitions
 *
 * @returns Extended Hono router
 *
 * @example
 * ```typescript
 * import { createRouter, websocket, sse, stream, cron } from '@agentuity/runtime';
 *
 * const router = createRouter();
 *
 * // Standard HTTP routes
 * router.get('/hello', (c) => c.text('Hello!'));
 * router.post('/data', async (c) => {
 *   const body = await c.req.json();
 *   return c.json({ received: body });
 * });
 *
 * // WebSocket connection
 * router.get('/ws', websocket((c, ws) => {
 *   ws.onMessage((event) => {
 *     ws.send('Echo: ' + event.data);
 *   });
 * }));
 *
 * // Server-Sent Events
 * router.get('/events', sse((c, stream) => {
 *   stream.writeSSE({ data: 'Hello', event: 'message' });
 * }));
 *
 * // Streaming response
 * router.post('/stream', stream((c) => {
 *   return new ReadableStream({
 *     start(controller) {
 *       controller.enqueue('data\n');
 *       controller.close();
 *     }
 *   });
 * }));
 *
 * // Cron job
 * router.post('/daily', cron('0 0 * * *', (c) => {
 *   return { status: 'complete' };
 * }));
 * ```
 */
export declare const createRouter: <E extends Env = Env, S extends Schema = Schema>() => Hono<E, S>;
//# sourceMappingURL=router.d.ts.map