import { FetchAdapter } from './adapter';
/**
 * Base properties shared by all vector upsert operations
 */
export interface VectorUpsertBase {
    /**
     * the key of the vector object which can be used as a reference. the value of this key is opaque to the vector storage.
     */
    key: string;
    /**
     * the metadata to upsert
     */
    metadata?: Record<string, unknown>;
}
/**
 * Upsert using pre-computed embeddings
 */
export interface VectorUpsertEmbeddings extends VectorUpsertBase {
    /**
     * the embeddings to upsert
     */
    embeddings: Array<number>;
    document?: never;
}
/**
 * Upsert using text that will be converted to embeddings
 */
export interface VectorUpsertText extends VectorUpsertBase {
    /**
     * the text to use for the embedding
     */
    document: string;
    embeddings?: never;
}
/**
 * Parameters for upserting a vector
 */
export type VectorUpsertParams = VectorUpsertEmbeddings | VectorUpsertText;
/**
 * Parameters for searching vectors
 */
export interface VectorSearchParams<T extends Record<string, unknown> = Record<string, unknown>> {
    /**
     * The text query to search for in the vector storage. This will be converted to embeddings
     * and used to find semantically similar documents.
     *
     * @example "comfortable office chair"
     * @example "machine learning algorithms"
     */
    query: string;
    /**
     * Maximum number of search results to return. If not specified, the server default will be used.
     * Must be a positive integer.
     *
     * @default 10
     * @example 5
     * @example 20
     */
    limit?: number;
    /**
     * Minimum similarity threshold for results. Only vectors with similarity scores greater than or equal
     * to this value will be returned. Value must be between 0.0 and 1.0, where 1.0 means exact match
     * and 0.0 means no similarity requirement.
     *
     * @minimum 0.0
     * @maximum 1.0
     * @example 0.7
     * @example 0.5
     */
    similarity?: number;
    /**
     * Metadata filters to apply to the search. Only vectors whose metadata matches all specified
     * key-value pairs will be included in results. Must be a valid JSON object if provided.
     *
     * @example { category: "furniture", inStock: true }
     * @example { userId: "123", type: "product" }
     */
    metadata?: T;
}
/**
 * Result of a vector search operation with optional type-safe metadata
 */
export interface VectorSearchResult<T extends Record<string, unknown> = Record<string, unknown>> {
    /**
     * the unique id of the object in vector storage
     */
    id: string;
    /**
     * the key used when the vector object was added to vector storage
     */
    key: string;
    /**
     * the metadata of the vector object when it was stored
     */
    metadata?: T;
    /**
     * the distance of the vector object from the query from 0-1. The larger the number, the more similar the vector object is to the query.
     */
    similarity: number;
}
/**
 * Extended search result that includes the document and embeddings
 */
export interface VectorSearchResultWithDocument<T extends Record<string, unknown> = Record<string, unknown>> extends VectorSearchResult<T> {
    /**
     * the document that was used to create the vector object
     */
    document?: string;
    /**
     * the embeddings of the vector object
     */
    embeddings?: Array<number>;
}
/**
 * Result of a vector upsert operation
 */
export interface VectorUpsertResult {
    /**
     * the key from the original upsert document
     */
    key: string;
    /**
     * the generated id for this vector in storage
     */
    id: string;
}
/**
 * Result when a vector is found by key
 */
export interface VectorResultFound<T extends Record<string, unknown> = Record<string, unknown>> {
    /**
     * the vector data
     */
    data: VectorSearchResultWithDocument<T>;
    /**
     * the vector was found
     */
    exists: true;
}
/**
 * Result when a vector is not found by key
 */
export interface VectorResultNotFound {
    /**
     * no data available
     */
    data: never;
    /**
     * the vector was not found
     */
    exists: false;
}
/**
 * Result of a get operation
 */
export type VectorResult<T extends Record<string, unknown> = Record<string, unknown>> = VectorResultFound<T> | VectorResultNotFound;
/**
 * Statistics for a vector namespace
 */
export interface VectorNamespaceStats {
    /**
     * Total size in bytes
     */
    sum: number;
    /**
     * Number of vectors in the namespace
     */
    count: number;
    /**
     * Unix timestamp (milliseconds) when the namespace was created
     */
    createdAt?: number;
    /**
     * Unix timestamp (milliseconds) when the namespace was last used
     */
    lastUsed?: number;
}
/**
 * Statistics for an individual vector item
 */
export interface VectorItemStats {
    /**
     * The embedding vector
     */
    embedding?: number[];
    /**
     * The original document text
     */
    document?: string;
    /**
     * Size in bytes
     */
    size: number;
    /**
     * Metadata associated with the vector
     */
    metadata?: Record<string, unknown>;
    /**
     * Unix timestamp (milliseconds) when the vector was first accessed
     */
    firstUsed: number;
    /**
     * Unix timestamp (milliseconds) when the vector was last accessed
     */
    lastUsed: number;
    /**
     * Number of times the vector was accessed.
     * Note: This is only tracked in cloud storage; local development returns undefined.
     */
    count?: number;
}
/**
 * Namespace statistics with sampled vector results
 */
export interface VectorNamespaceStatsWithSamples extends VectorNamespaceStats {
    /**
     * A sample of vectors in the namespace (up to 20)
     */
    sampledResults?: Record<string, VectorItemStats>;
}
/**
 * Parameters for getting all namespace statistics with optional pagination
 */
export interface VectorGetAllStatsParams {
    /**
     * Maximum number of namespaces to return (default: 100, max: 1000)
     */
    limit?: number;
    /**
     * Number of namespaces to skip for pagination (default: 0)
     */
    offset?: number;
}
/**
 * Paginated response for vector namespace statistics
 */
export interface VectorStatsPaginated {
    /**
     * Map of namespace names to their statistics
     */
    namespaces: Record<string, VectorNamespaceStats>;
    /**
     * Total number of namespaces across all pages
     */
    total: number;
    /**
     * Number of namespaces requested per page
     */
    limit: number;
    /**
     * Number of namespaces skipped
     */
    offset: number;
    /**
     * Whether there are more namespaces available
     */
    hasMore: boolean;
}
/**
 * Vector storage service for managing vector embeddings and semantic search
 */
export interface VectorStorage {
    /**
     * Upsert vectors into the vector storage. You can provide either pre-computed embeddings
     * or text documents that will be automatically embedded.
     *
     * @param name - the name of the vector storage namespace
     * @param documents - one or more documents to upsert
     * @returns array of results with key-to-id mappings for the upserted vectors
     *
     * @example
     * ```typescript
     * // Upsert with automatic embedding
     * const results = await vectorStore.upsert('products',
     *   {
     *     key: 'chair-001',
     *     document: 'Comfortable office chair with lumbar support',
     *     metadata: { category: 'furniture', price: 299 }
     *   },
     *   {
     *     key: 'desk-001',
     *     document: 'Standing desk with adjustable height',
     *     metadata: { category: 'furniture', price: 599 }
     *   }
     * );
     *
     * // Upsert with pre-computed embeddings
     * await vectorStore.upsert('products', {
     *   key: 'lamp-001',
     *   embeddings: [0.1, 0.2, 0.3, ...], // Your pre-computed vector
     *   metadata: { category: 'lighting', price: 49 }
     * });
     * ```
     */
    upsert(name: string, ...documents: VectorUpsertParams[]): Promise<VectorUpsertResult[]>;
    /**
     * Get a single vector by its key
     *
     * @param name - the name of the vector storage namespace
     * @param key - the key of the vector to retrieve
     * @returns VectorResult with exists field for type-safe checking
     *
     * @example
     * ```typescript
     * const result = await vectorStore.get('products', 'chair-001');
     * if (result.exists) {
     *   console.log('Found:', result.data.metadata);
     *   console.log('Embeddings:', result.data.embeddings);
     * }
     * ```
     */
    get<T extends Record<string, unknown> = Record<string, unknown>>(name: string, key: string): Promise<VectorResult<T>>;
    /**
     * Get multiple vectors by their keys in a single request
     *
     * @param name - the name of the vector storage namespace
     * @param keys - the keys of the vectors to retrieve
     * @returns a Map of key to vector data (only includes found vectors)
     *
     * @example
     * ```typescript
     * const vectors = await vectorStore.getMany('products', 'chair-001', 'desk-001', 'lamp-001');
     * for (const [key, data] of vectors) {
     *   console.log(`${key}:`, data.metadata);
     * }
     * ```
     */
    getMany<T extends Record<string, unknown> = Record<string, unknown>>(name: string, ...keys: string[]): Promise<Map<string, VectorSearchResultWithDocument<T>>>;
    /**
     * Search for semantically similar vectors using natural language queries
     *
     * @param name - the name of the vector storage namespace
     * @param params - search parameters including query, filters, and limits
     * @returns array of search results ordered by similarity (highest first)
     *
     * @example
     * ```typescript
     * // Basic semantic search
     * const results = await vectorStore.search('products', {
     *   query: 'comfortable seating for office',
     *   limit: 5
     * });
     *
     * // Search with metadata filters and similarity threshold
     * const filteredResults = await vectorStore.search('products', {
     *   query: 'ergonomic furniture',
     *   limit: 10,
     *   similarity: 0.7, // Only results with 70%+ similarity
     *   metadata: { category: 'furniture', inStock: true }
     * });
     *
     * for (const result of filteredResults) {
     *   console.log(`${result.key}: ${result.similarity * 100}% match`);
     *   console.log('Metadata:', result.metadata);
     * }
     * ```
     */
    search<T extends Record<string, unknown> = Record<string, unknown>>(name: string, params: VectorSearchParams<T>): Promise<VectorSearchResult<T>[]>;
    /**
     * Delete one or more vectors by their keys
     *
     * @param name - the name of the vector storage namespace
     * @param keys - one or more keys of vectors to delete
     * @returns the number of vectors that were deleted
     *
     * @example
     * ```typescript
     * // Delete a single vector
     * const deleted = await vectorStore.delete('products', 'chair-001');
     * console.log(`Deleted ${deleted} vector(s)`);
     *
     * // Delete multiple vectors
     * await vectorStore.delete('products', 'chair-001', 'desk-001', 'lamp-001');
     * ```
     */
    delete(name: string, ...keys: string[]): Promise<number>;
    /**
     * Check if a vector storage namespace exists
     *
     * @param name - the name of the vector storage namespace
     * @returns true if the namespace exists, false otherwise
     *
     * @example
     * ```typescript
     * if (await vectorStore.exists('products')) {
     *   console.log('Products namespace exists');
     * }
     * ```
     */
    exists(name: string): Promise<boolean>;
    /**
     * Get statistics for a specific namespace including sampled results
     *
     * @param name - the name of the vector storage namespace
     * @returns statistics for the namespace with sampled vector results
     *
     * @example
     * ```typescript
     * const stats = await vectorStore.getStats('products');
     * console.log(`Namespace has ${stats.count} vectors (${stats.sum} bytes)`);
     * if (stats.sampledResults) {
     *   for (const [key, item] of Object.entries(stats.sampledResults)) {
     *     console.log(`${key}: ${item.size} bytes, accessed ${item.count} times`);
     *   }
     * }
     * ```
     */
    getStats(name: string): Promise<VectorNamespaceStatsWithSamples>;
    /**
     * get statistics for all namespaces with optional pagination
     *
     * @param params - optional pagination parameters (limit, offset)
     * @returns map of namespace names to statistics, or paginated response if params provided
     *
     * @remarks
     * - Without params: returns flat map of all namespaces (backward compatible)
     * - With params: returns paginated response with total count and hasMore flag
     * - Default limit is 100, maximum is 1000
     */
    getAllStats(params?: VectorGetAllStatsParams): Promise<Record<string, VectorNamespaceStats> | VectorStatsPaginated>;
    /**
     * Get all namespace names
     *
     * @returns array of namespace names (up to 1000)
     *
     * @remarks
     * This method returns a maximum of 1000 namespace names.
     * If you have more than 1000 namespaces, only the first 1000
     * (ordered by creation date, most recent first) will be returned.
     *
     * @example
     * ```typescript
     * const namespaces = await vectorStore.getNamespaces();
     * console.log('Namespaces:', namespaces.join(', '));
     * ```
     */
    getNamespaces(): Promise<string[]>;
    /**
     * Delete an entire namespace and all its vectors
     *
     * @param name - the name of the vector storage namespace to delete
     *
     * @example
     * ```typescript
     * await vectorStore.deleteNamespace('old-products');
     * console.log('Namespace deleted');
     * ```
     */
    deleteNamespace(name: string): Promise<void>;
}
export declare class VectorStorageService implements VectorStorage {
    #private;
    constructor(baseUrl: string, adapter: FetchAdapter);
    upsert(name: string, ...documents: VectorUpsertParams[]): Promise<VectorUpsertResult[]>;
    get<T extends Record<string, unknown> = Record<string, unknown>>(name: string, key: string): Promise<VectorResult<T>>;
    getMany<T extends Record<string, unknown> = Record<string, unknown>>(name: string, ...keys: string[]): Promise<Map<string, VectorSearchResultWithDocument<T>>>;
    search<T extends Record<string, unknown> = Record<string, unknown>>(name: string, params: VectorSearchParams<T>): Promise<VectorSearchResult<T>[]>;
    delete(name: string, ...keys: string[]): Promise<number>;
    exists(name: string): Promise<boolean>;
    getStats(name: string): Promise<VectorNamespaceStatsWithSamples>;
    getAllStats(params?: VectorGetAllStatsParams): Promise<Record<string, VectorNamespaceStats> | VectorStatsPaginated>;
    getNamespaces(): Promise<string[]>;
    deleteNamespace(name: string): Promise<void>;
}
//# sourceMappingURL=vector.d.ts.map