import { z } from 'zod';
import { APIResponseSchema } from '../api';
import { SandboxResponseError, API_VERSION } from './util';
const SandboxResourcesSchema = z
    .object({
    memory: z.string().optional().describe('Memory limit (e.g., "512Mi", "1Gi")'),
    cpu: z.string().optional().describe('CPU limit in millicores (e.g., "500m", "1000m")'),
    disk: z.string().optional().describe('Disk limit (e.g., "1Gi", "10Gi")'),
})
    .describe('Resource limits for the sandbox');
const SandboxUserInfoSchema = z
    .object({
    id: z.string().describe('User ID'),
    firstName: z.string().optional().describe("User's first name"),
    lastName: z.string().optional().describe("User's last name"),
})
    .describe('User who created the sandbox');
const SandboxAgentInfoSchema = z
    .object({
    id: z.string().describe('Agent ID'),
    name: z.string().describe('Agent name'),
})
    .describe('Agent associated with the sandbox');
const SandboxProjectInfoSchema = z
    .object({
    id: z.string().describe('Project ID'),
    name: z.string().describe('Project name'),
})
    .describe('Project associated with the sandbox');
const SandboxOrgInfoSchema = z
    .object({
    id: z.string().describe('Organization ID'),
    name: z.string().describe('Organization name'),
})
    .describe('Organization associated with the sandbox');
const SandboxInfoDataSchema = z
    .object({
    sandboxId: z.string().describe('Unique identifier for the sandbox'),
    identifier: z.string().optional().describe('Short identifier for DNS hostname'),
    name: z.string().optional().describe('Sandbox name'),
    description: z.string().optional().describe('Sandbox description'),
    status: z
        .enum(['creating', 'idle', 'running', 'terminated', 'failed', 'deleted'])
        .describe('Current status of the sandbox'),
    mode: z.string().optional().describe('Sandbox mode (interactive or oneshot)'),
    createdAt: z.string().describe('ISO timestamp when the sandbox was created'),
    region: z.string().optional().describe('Region where the sandbox is running'),
    runtimeId: z.string().optional().describe('Runtime ID'),
    runtimeName: z.string().optional().describe('Runtime name (e.g., "bun:1")'),
    runtimeIconUrl: z.string().optional().describe('URL for runtime icon'),
    snapshotId: z.string().optional().describe('Snapshot ID this sandbox was created from'),
    snapshotTag: z.string().optional().describe('Snapshot tag this sandbox was created from'),
    executions: z.number().describe('Total number of executions in this sandbox'),
    exitCode: z
        .number()
        .optional()
        .describe('Exit code from the last execution (only for terminated/failed sandboxes)'),
    stdoutStreamUrl: z.string().optional().describe('URL for streaming stdout output'),
    stderrStreamUrl: z.string().optional().describe('URL for streaming stderr output'),
    dependencies: z
        .array(z.string())
        .optional()
        .describe('Apt packages installed in the sandbox'),
    metadata: z
        .record(z.string(), z.unknown())
        .optional()
        .describe('User-defined metadata associated with the sandbox'),
    resources: SandboxResourcesSchema.optional().describe('Resource limits for this sandbox'),
    cpuTimeMs: z.number().optional().describe('Total CPU time consumed in milliseconds'),
    memoryByteSec: z.number().optional().describe('Total memory usage in byte-seconds'),
    networkEgressBytes: z.number().optional().describe('Total network egress in bytes'),
    networkEnabled: z.boolean().optional().describe('Whether network access is enabled'),
    networkPort: z.number().optional().describe('Network port exposed from the sandbox'),
    url: z
        .string()
        .optional()
        .describe('Public URL for the sandbox (only set if networkPort is configured)'),
    user: SandboxUserInfoSchema.optional().describe('User who created the sandbox'),
    agent: SandboxAgentInfoSchema.optional().describe('Agent associated with the sandbox'),
    project: SandboxProjectInfoSchema.optional().describe('Project associated with the sandbox'),
    org: SandboxOrgInfoSchema.describe('Organization associated with the sandbox'),
})
    .describe('Detailed information about a sandbox');
const SandboxGetResponseSchema = APIResponseSchema(SandboxInfoDataSchema);
/**
 * Retrieves information about a specific sandbox.
 *
 * @param client - The API client to use for the request
 * @param params - Parameters including the sandbox ID
 * @returns Sandbox information including status, creation time, and execution count
 * @throws {SandboxResponseError} If the sandbox is not found or request fails
 */
export async function sandboxGet(client, params) {
    const { sandboxId, orgId, includeDeleted } = params;
    const queryParams = new URLSearchParams();
    if (orgId) {
        queryParams.set('orgId', orgId);
    }
    if (includeDeleted) {
        queryParams.set('includeDeleted', 'true');
    }
    const queryString = queryParams.toString();
    const url = `/sandbox/${API_VERSION}/${sandboxId}${queryString ? `?${queryString}` : ''}`;
    const resp = await client.get(url, SandboxGetResponseSchema);
    if (resp.success) {
        return {
            sandboxId: resp.data.sandboxId,
            identifier: resp.data.identifier,
            name: resp.data.name,
            description: resp.data.description,
            status: resp.data.status,
            mode: resp.data.mode,
            createdAt: resp.data.createdAt,
            region: resp.data.region,
            runtimeId: resp.data.runtimeId,
            runtimeName: resp.data.runtimeName,
            runtimeIconUrl: resp.data.runtimeIconUrl,
            snapshotId: resp.data.snapshotId,
            snapshotTag: resp.data.snapshotTag,
            executions: resp.data.executions,
            exitCode: resp.data.exitCode,
            stdoutStreamUrl: resp.data.stdoutStreamUrl,
            stderrStreamUrl: resp.data.stderrStreamUrl,
            dependencies: resp.data.dependencies,
            metadata: resp.data.metadata,
            resources: resp.data.resources,
            cpuTimeMs: resp.data.cpuTimeMs,
            memoryByteSec: resp.data.memoryByteSec,
            networkEgressBytes: resp.data.networkEgressBytes,
            networkEnabled: resp.data.networkEnabled,
            networkPort: resp.data.networkPort,
            url: resp.data.url,
            user: resp.data.user,
            agent: resp.data.agent,
            project: resp.data.project,
            org: resp.data.org,
        };
    }
    throw new SandboxResponseError({ message: resp.message, sandboxId });
}
//# sourceMappingURL=get.js.map