import { formatMessage } from './util';
const cyan = '\x1b[1;96m';
const reset = '\x1b[0m';
/**
 * Simple internal logger that doesn't depend on other SDK modules
 * This logger is only for SDK internal diagnostics and debugging
 */
class InternalLogger {
    config;
    constructor() {
        this.config = this.loadConfig();
    }
    /**
     * Load configuration from environment variables
     */
    loadConfig() {
        const envLevel = process.env.AGENTUITY_SDK_LOG_LEVEL?.toLowerCase();
        // Validate log level
        const validLevels = ['debug', 'info', 'warn', 'error', 'silent'];
        const level = validLevels.includes(envLevel) ? envLevel : 'silent';
        return {
            level,
            context: {
                '@agentuity/source': 'sdk-internal',
                '@agentuity/timestamp': new Date().toISOString(),
            },
        };
    }
    /**
     * Check if a log level should be output based on current configuration
     */
    shouldLog(level) {
        if (this.config.level === 'silent')
            return false;
        const levelPriority = {
            debug: 0,
            info: 1,
            warn: 2,
            error: 3,
            silent: 4,
        };
        return levelPriority[level] >= levelPriority[this.config.level];
    }
    /**
     * Format a log message with context
     */
    formatMessage(message, ...args) {
        const formattedMessage = formatMessage(false, this.config.context, message, args);
        return `${cyan}[INTERNAL]${reset} ${formattedMessage}`;
    }
    /**
     * Log a debug message
     */
    debug(message, ...args) {
        if (this.shouldLog('debug')) {
            console.debug(this.formatMessage(message, ...args));
        }
    }
    /**
     * Log an info message
     */
    info(message, ...args) {
        if (this.shouldLog('info')) {
            console.info(this.formatMessage(message, ...args));
        }
    }
    /**
     * Log a warning message
     */
    warn(message, ...args) {
        if (this.shouldLog('warn')) {
            console.warn(this.formatMessage(message, ...args));
        }
    }
    /**
     * Log an error message
     */
    error(message, ...args) {
        if (this.shouldLog('error')) {
            console.error(this.formatMessage(message, ...args));
        }
    }
    /**
     * Update configuration at runtime
     */
    updateConfig(config) {
        this.config = { ...this.config, ...config };
    }
    /**
     * Get current configuration
     */
    getConfig() {
        return { ...this.config };
    }
    /**
     * Check if logging is enabled
     */
    isEnabled() {
        return this.config.level !== 'silent';
    }
    /**
     * Create a child logger with additional context
     */
    child(context) {
        const childLogger = new InternalLogger();
        childLogger.updateConfig({
            ...this.config,
            context: {
                ...this.config.context,
                ...context,
            },
        });
        return childLogger;
    }
}
// Singleton instance - not exported
const internalLogger = new InternalLogger();
/**
 * Internal logger for SDK use only
 * This is NOT exported from the main SDK index
 */
export const internal = {
    debug: (message, ...args) => internalLogger.debug(message, ...args),
    info: (message, ...args) => internalLogger.info(message, ...args),
    warn: (message, ...args) => internalLogger.warn(message, ...args),
    error: (message, ...args) => internalLogger.error(message, ...args),
    // Utility methods
    updateConfig: (config) => internalLogger.updateConfig(config),
    getConfig: () => internalLogger.getConfig(),
    isEnabled: () => internalLogger.isEnabled(),
    child: (context) => internalLogger.child(context),
};
//# sourceMappingURL=internal.js.map