/**
 * @module validation
 * Queue validation utilities with constants matching the Catalyst backend.
 *
 * These validation functions perform client-side validation before API calls,
 * providing immediate feedback and reducing unnecessary network requests.
 */
/** Maximum allowed length for queue names. */
export declare const MAX_QUEUE_NAME_LENGTH = 256;
/** Minimum allowed length for queue names. */
export declare const MIN_QUEUE_NAME_LENGTH = 1;
/** Maximum payload size in bytes (1MB). */
export declare const MAX_PAYLOAD_SIZE = 1048576;
/** Maximum description length in characters. */
export declare const MAX_DESCRIPTION_LENGTH = 1024;
/** Maximum number of messages in a single batch operation. */
export declare const MAX_BATCH_SIZE = 1000;
/** Maximum metadata size in bytes (64KB). */
export declare const MAX_METADATA_SIZE = 65536;
/** Maximum partition key length in characters. */
export declare const MAX_PARTITION_KEY_LENGTH = 256;
/** Maximum idempotency key length in characters. */
export declare const MAX_IDEMPOTENCY_KEY_LENGTH = 256;
/** Maximum visibility timeout in seconds (12 hours). */
export declare const MAX_VISIBILITY_TIMEOUT = 43200;
/** Maximum number of retry attempts allowed. */
export declare const MAX_RETRIES = 100;
/** Maximum number of in-flight messages per client. */
export declare const MAX_IN_FLIGHT = 1000;
/** Maximum source name length. */
export declare const MAX_SOURCE_NAME_LENGTH = 256;
/**
 * Error thrown when validation fails for queue operations.
 *
 * Includes the field name and optionally the invalid value for debugging.
 *
 * @example
 * ```typescript
 * try {
 *   validateQueueName('Invalid Name!');
 * } catch (error) {
 *   if (error instanceof QueueValidationError) {
 *     console.error(`Invalid ${error.field}: ${error.message}`);
 *   }
 * }
 * ```
 */
export declare const QueueValidationError: {
    new (args?: ({
        field: string;
        value?: unknown;
    } & {
        message?: string;
        cause?: unknown;
    }) | undefined): import("@agentuity/core").RichError & {
        readonly _tag: "QueueValidationError";
    } & Readonly<{
        field: string;
        value?: unknown;
    }>;
    readonly defaultMessage?: string;
};
/**
 * Validates a queue name against naming rules.
 *
 * Queue names must:
 * - Be 1-256 characters long
 * - Start with a lowercase letter or underscore
 * - Contain only lowercase letters, digits, underscores, and hyphens
 *
 * @param name - The queue name to validate
 * @throws {QueueValidationError} If the name is invalid
 *
 * @example
 * ```typescript
 * validateQueueName('my_queue');      // OK
 * validateQueueName('order-queue');   // OK
 * validateQueueName('Invalid Name!'); // Throws
 * ```
 */
export declare function validateQueueName(name: string): void;
/**
 * Validates a queue type.
 *
 * @param type - The queue type to validate
 * @throws {QueueValidationError} If the type is not 'worker' or 'pubsub'
 */
export declare function validateQueueType(type: string): void;
/**
 * Validates a message payload.
 *
 * Payloads must be non-empty JSON and not exceed 1MB when serialized.
 *
 * @param payload - The payload to validate (must be JSON-serializable)
 * @throws {QueueValidationError} If the payload is empty or too large
 */
export declare function validatePayload(payload: unknown): void;
/**
 * Validates a message ID format.
 *
 * Message IDs must start with the `qmsg_` prefix.
 *
 * @param id - The message ID to validate
 * @throws {QueueValidationError} If the ID format is invalid
 */
export declare function validateMessageId(id: string): void;
/**
 * Validates a destination ID format.
 *
 * Destination IDs must start with the `qdest_` prefix.
 *
 * @param id - The destination ID to validate
 * @throws {QueueValidationError} If the ID format is invalid
 */
export declare function validateDestinationId(id: string): void;
/**
 * Validates a queue or message description.
 *
 * @param description - The description to validate (optional)
 * @throws {QueueValidationError} If the description exceeds the maximum length
 */
export declare function validateDescription(description?: string): void;
/**
 * Validates a partition key length.
 *
 * Partition keys are used for message ordering within a queue.
 *
 * @param key - The partition key to validate (optional)
 * @throws {QueueValidationError} If the key exceeds the maximum length
 */
export declare function validatePartitionKey(key?: string): void;
/**
 * Validates an idempotency key length.
 *
 * Idempotency keys are used to prevent duplicate message processing.
 *
 * @param key - The idempotency key to validate (optional)
 * @throws {QueueValidationError} If the key exceeds the maximum length
 */
export declare function validateIdempotencyKey(key?: string): void;
/**
 * Validates a time-to-live value.
 *
 * TTL specifies how long a message should be kept before expiring.
 *
 * @param ttl - The TTL in seconds to validate (optional)
 * @throws {QueueValidationError} If the TTL is negative
 */
export declare function validateTTL(ttl?: number): void;
/**
 * Validates a visibility timeout.
 *
 * Visibility timeout is how long a message is hidden after being received,
 * giving the consumer time to process it before it becomes visible again.
 *
 * @param timeout - The timeout in seconds to validate (optional)
 * @throws {QueueValidationError} If the timeout is out of valid range (1-43200 seconds)
 */
export declare function validateVisibilityTimeout(timeout?: number): void;
/**
 * Validates the maximum retry count.
 *
 * @param retries - The max retries value to validate (optional)
 * @throws {QueueValidationError} If retries is negative or exceeds maximum
 */
export declare function validateMaxRetries(retries?: number): void;
/**
 * Validates the maximum in-flight messages per client.
 *
 * This controls how many messages a single consumer can process concurrently.
 *
 * @param maxInFlight - The max in-flight value to validate (optional)
 * @throws {QueueValidationError} If the value is out of valid range (1-1000)
 */
export declare function validateMaxInFlight(maxInFlight?: number): void;
/**
 * Validates a message offset.
 *
 * Offsets are sequential positions in the queue, starting from 0.
 *
 * @param offset - The offset value to validate
 * @throws {QueueValidationError} If the offset is negative
 */
export declare function validateOffset(offset: number): void;
/**
 * Validates a limit for list/consume operations.
 *
 * @param limit - The limit value to validate
 * @throws {QueueValidationError} If the limit is less than 1 or exceeds maximum
 */
export declare function validateLimit(limit: number): void;
/**
 * Validates a batch size for batch operations.
 *
 * @param size - The batch size to validate
 * @throws {QueueValidationError} If the size is less than 1 or exceeds maximum
 */
export declare function validateBatchSize(size: number): void;
/**
 * Validates a webhook URL for destinations.
 *
 * URLs must use HTTP or HTTPS protocol.
 *
 * @param url - The URL to validate
 * @throws {QueueValidationError} If the URL is missing or not HTTP/HTTPS
 */
export declare function validateWebhookUrl(url: string): void;
/**
 * Validates a destination configuration object.
 *
 * Checks that the config contains a valid URL and optional method/timeout settings.
 *
 * @param config - The destination config object to validate
 * @throws {QueueValidationError} If the config is invalid
 */
export declare function validateDestinationConfig(config: Record<string, unknown>): void;
/**
 * Validates a source ID format.
 *
 * Source IDs must start with the `qsrc_` prefix.
 *
 * @param id - The source ID to validate
 * @throws {QueueValidationError} If the ID format is invalid
 *
 * @example
 * ```typescript
 * validateSourceId('qsrc_abc123'); // OK
 * validateSourceId('invalid');     // Throws
 * ```
 */
export declare function validateSourceId(id: string): void;
/**
 * Validates a source name.
 *
 * Source names must be non-empty and not exceed the maximum length.
 *
 * @param name - The source name to validate
 * @throws {QueueValidationError} If the name is invalid
 *
 * @example
 * ```typescript
 * validateSourceName('my-source');   // OK
 * validateSourceName('');          // Throws
 * ```
 */
export declare function validateSourceName(name: string): void;
//# sourceMappingURL=validation.d.ts.map