import type { StandardSchemaV1 } from '@agentuity/core';
/**
 * Symbol used to identify schema types in a minification-safe way.
 * Uses Symbol.for() to ensure the same symbol is used across bundled modules.
 */
export declare const SCHEMA_KIND: unique symbol;
/**
 * A validation issue from a failed schema validation.
 */
export type ValidationIssue = StandardSchemaV1.Issue;
/**
 * The result of a schema validation (success or failure).
 */
export type ValidationResult<T> = StandardSchemaV1.Result<T>;
/**
 * Successful parse result containing validated data.
 */
export interface SafeParseSuccess<T> {
    /** Indicates successful validation */
    success: true;
    /** The validated and typed data */
    data: T;
}
/**
 * Failed parse result containing validation error.
 */
export interface SafeParseError {
    /** Indicates failed validation */
    success: false;
    /** The validation error with detailed issues */
    error: ValidationError;
}
/**
 * Result of safeParse - either success with data or failure with error.
 */
export type SafeParseResult<T> = SafeParseSuccess<T> | SafeParseError;
/**
 * Error thrown when schema validation fails.
 * Contains detailed information about all validation issues including field paths.
 *
 * @example
 * ```typescript
 * try {
 *   schema.parse(data);
 * } catch (error) {
 *   if (error instanceof ValidationError) {
 *     console.log(error.message);  // Human-readable error
 *     console.log(error.issues);   // Detailed issues array
 *   }
 * }
 * ```
 */
export declare class ValidationError extends Error {
    /** Array of validation issues with paths and messages */
    readonly issues: readonly ValidationIssue[];
    constructor(issues: readonly ValidationIssue[]);
    toString(): string;
}
/**
 * Base schema interface that all schemas implement.
 * Provides StandardSchema v1 compliance plus additional methods for parsing and description.
 */
export interface Schema<Input = unknown, Output = Input> extends StandardSchemaV1<Input, Output> {
    readonly '~standard': StandardSchemaV1.Props<Input, Output>;
    /** Optional description for documentation */
    description?: string;
    /** Add a description to the schema for documentation and JSON Schema */
    describe(description: string): this;
    /** Parse and validate data, throwing ValidationError on failure */
    parse(value: unknown): Output;
    /** Parse and validate data, returning result object without throwing */
    safeParse(value: unknown): SafeParseResult<Output>;
    /** Make this schema optional (allow undefined) */
    optional(): Schema<Input | undefined, Output | undefined>;
    /** Make this schema nullable (allow null) */
    nullable(): Schema<Input | null, Output | null>;
}
/**
 * Extract the output type from a schema (like zod's z.infer).
 *
 * @example
 * ```typescript
 * const User = s.object({ name: s.string(), age: s.number() });
 * type User = Infer<typeof User>;  // { name: string; age: number }
 * ```
 */
export type Infer<T extends Schema<any, any>> = StandardSchemaV1.InferOutput<T>;
/**
 * Extract the input type from a schema.
 */
export type InferInput<T extends Schema<any, any>> = StandardSchemaV1.InferInput<T>;
/**
 * Extract the output type from a schema (alias for Infer).
 */
export type InferOutput<T extends Schema<any, any>> = StandardSchemaV1.InferOutput<T>;
/**
 * Create a validation issue with an optional field path.
 */
export declare function createIssue(message: string, path?: ReadonlyArray<PropertyKey | StandardSchemaV1.PathSegment>): ValidationIssue;
/**
 * Create a successful validation result.
 */
export declare function success<T>(value: T): StandardSchemaV1.SuccessResult<T>;
/**
 * Create a failed validation result.
 */
export declare function failure(issues: ValidationIssue[]): StandardSchemaV1.FailureResult;
/**
 * Create parse and safeParse methods for a schema.
 * @internal
 */
export declare function createParseMethods<Output>(): {
    parse(this: Schema<any, Output>, value: unknown): Output;
    safeParse(this: Schema<any, Output>, value: unknown): SafeParseResult<Output>;
};
//# sourceMappingURL=base.d.ts.map