import { Database } from 'bun:sqlite';
import { join } from 'node:path';
import { mkdir } from 'node:fs/promises';
import { getDefaultConfigDir } from '../config';
const TTL_DAYS = 7;
const TTL_MS = TTL_DAYS * 24 * 60 * 60 * 1000;
let db = null;
async function getDatabase() {
    if (db) {
        return db;
    }
    const configDir = getDefaultConfigDir();
    await mkdir(configDir, { recursive: true });
    const dbPath = join(configDir, 'resource.db');
    db = new Database(dbPath);
    db.run(`
		CREATE TABLE IF NOT EXISTS resource_region_cache (
			resource_type TEXT NOT NULL,
			profile TEXT NOT NULL,
			id TEXT NOT NULL,
			region TEXT NOT NULL,
			org_id TEXT,
			last_updated INTEGER NOT NULL,
			PRIMARY KEY (resource_type, profile, id)
		)
	`);
    db.run(`
		CREATE INDEX IF NOT EXISTS idx_last_updated 
		ON resource_region_cache(last_updated)
	`);
    return db;
}
function pruneOldEntries(database) {
    const cutoff = Date.now() - TTL_MS;
    database.run('DELETE FROM resource_region_cache WHERE last_updated < ?', [cutoff]);
}
/**
 * Get the cached info (region and orgId) for a resource.
 * Returns null if not found or expired.
 */
export async function getResourceInfo(type, profile, id) {
    const database = await getDatabase();
    const cutoff = Date.now() - TTL_MS;
    const row = database
        .query('SELECT region, org_id, last_updated FROM resource_region_cache WHERE resource_type = ? AND profile = ? AND id = ?')
        .get(type, profile, id);
    if (!row) {
        return null;
    }
    // Check if entry is expired
    if (row.last_updated < cutoff) {
        // Remove stale entry
        database.run('DELETE FROM resource_region_cache WHERE resource_type = ? AND profile = ? AND id = ?', [type, profile, id]);
        return null;
    }
    return {
        region: row.region,
        orgId: row.org_id ?? undefined,
    };
}
/**
 * Get the cached region for a resource.
 * Returns null if not found or expired.
 * @deprecated Use getResourceInfo() to get both region and orgId
 */
export async function getResourceRegion(type, profile, id) {
    const info = await getResourceInfo(type, profile, id);
    return info?.region ?? null;
}
/**
 * Set the cached info for a resource.
 * Uses INSERT OR REPLACE to upsert.
 */
export async function setResourceInfo(type, profile, id, region, orgId) {
    const database = await getDatabase();
    pruneOldEntries(database);
    database.run(`INSERT OR REPLACE INTO resource_region_cache 
		 (resource_type, profile, id, region, org_id, last_updated)
		 VALUES (?, ?, ?, ?, ?, ?)`, [type, profile, id, region, orgId ?? null, Date.now()]);
}
/**
 * Set the cached region for a resource.
 * Uses INSERT OR REPLACE to upsert.
 * @deprecated Use setResourceInfo() to set both region and orgId
 */
export async function setResourceRegion(type, profile, id, region) {
    await setResourceInfo(type, profile, id, region);
}
/**
 * Delete the cached info for a resource.
 * Called when a resource is deleted.
 */
export async function deleteResourceRegion(type, profile, id) {
    const database = await getDatabase();
    database.run('DELETE FROM resource_region_cache WHERE resource_type = ? AND profile = ? AND id = ?', [type, profile, id]);
}
/**
 * Clear all cached entries for a specific profile.
 * Useful when switching profiles or logging out.
 */
export async function clearProfileCache(profile) {
    const database = await getDatabase();
    database.run('DELETE FROM resource_region_cache WHERE profile = ?', [profile]);
}
/**
 * Close the database connection.
 * Should be called on CLI exit for clean shutdown.
 */
export function closeDatabase() {
    if (db) {
        db.close();
        db = null;
    }
}
//# sourceMappingURL=resource-region.js.map