import { z } from 'zod';
import { APIResponseSchema } from '../api';
import { RegionResponseError } from './util';
const DeleteResourceSpec = z.object({
    type: z.enum(['db', 's3']).describe('the resource type'),
    name: z.string().describe('the resource name (bucket_name for S3, db_name for DB)'),
});
const DeleteResourcesRequest = z.object({
    resources: z.array(DeleteResourceSpec).describe('list of resources to delete'),
});
const DeletedResource = z.object({
    type: z.string().describe('the resource type'),
    name: z.string().describe('the resource name'),
    env_keys: z.array(z.string()).describe('environment variable keys to remove'),
});
const DeleteResourcesResponse = z.object({
    deleted: z.array(DeletedResource).describe('list of deleted resources'),
});
const DeleteResourcesResponseSchema = APIResponseSchema(DeleteResourcesResponse);
/**
 * Delete one or more resources (DB or S3) for an organization in a specific region
 * Requires CLI authentication
 *
 * @param client - Catalyst API client
 * @param orgId - Organization ID
 * @param region - Cloud region
 * @param resources - Array of resources to delete
 * @returns
 */
export async function deleteResources(client, orgId, region, resources) {
    const resp = await client.request('DELETE', `/resource/2025-11-16/${orgId}/${region}`, DeleteResourcesResponseSchema, { resources }, DeleteResourcesRequest);
    if (resp.success) {
        return resp.data.deleted;
    }
    throw new RegionResponseError({ message: resp.message });
}
//# sourceMappingURL=delete.js.map