import { z } from 'zod';
// Schema for the delegate tool
export const DelegateArgsSchema = z.object({
    agent: z
        .enum(['scout', 'builder', 'reviewer', 'memory', 'expert'])
        .describe('The agent to delegate to'),
    task: z.string().describe('Clear description of the task to delegate'),
    context: z.string().optional().describe('Additional context from previous tasks'),
    waitForResult: z
        .boolean()
        .default(true)
        .describe('Whether to wait for the result before continuing'),
});
// Agent display names for @mentions
const AGENT_MENTIONS = {
    lead: '@Agentuity Coder Lead',
    scout: '@Agentuity Coder Scout',
    builder: '@Agentuity Coder Builder',
    reviewer: '@Agentuity Coder Reviewer',
    memory: '@Agentuity Coder Memory',
    expert: '@Agentuity Coder Expert',
};
export const delegateTool = {
    name: 'coder_delegate',
    description: `Delegate a task to a specialized Agentuity Coder agent.

Use this to:
- Scout: Explore codebase, find patterns, research documentation
- Builder: Implement features, write code, run tests
- Reviewer: Review changes, catch issues, apply fixes  
- Memory: Store context, remember decisions across sessions
- Expert: Get help with Agentuity CLI and cloud services

The task will be executed by the specified agent and the result returned.`,
    args: DelegateArgsSchema,
    async execute(args, _context) {
        const mention = AGENT_MENTIONS[args.agent];
        // Build the delegation prompt
        let prompt = `${mention}\n\n## Task\n${args.task}`;
        if (args.context) {
            prompt = `${mention}\n\n## Context\n${args.context}\n\n## Task\n${args.task}`;
        }
        // In Open Code, this would trigger the Task tool with the appropriate agent
        // For now, return the formatted prompt that Lead should use
        return {
            output: `To delegate this task, use the Task tool with this prompt:\n\n${prompt}\n\nThe ${args.agent} agent will handle this task.`,
        };
    },
};
export default delegateTool;
//# sourceMappingURL=delegate.js.map