import type { Context, Handler } from 'hono';
import type { Env } from '../app';
/**
 * Handler function for cron jobs.
 * Receives the Hono context and can return any response.
 */
export type CronHandler<E extends Env = Env> = (c: Context<E>) => unknown | Promise<unknown>;
/**
 * Creates a cron middleware for scheduled task endpoints.
 *
 * **Important:** Cron endpoints must use POST method. The middleware will throw
 * an error if called with any other HTTP method.
 *
 * Use with router.post() to create a cron endpoint:
 *
 * @example
 * ```typescript
 * import { createRouter, cron } from '@agentuity/runtime';
 *
 * const router = createRouter();
 *
 * // Daily cleanup at midnight
 * router.post('/daily-cleanup', cron('0 0 * * *', (c) => {
 *   c.var.logger.info('Running daily cleanup');
 *   return { status: 'cleanup complete' };
 * }));
 *
 * // Hourly health check
 * router.post('/health-check', cron('0 * * * *', (c) => {
 *   c.var.logger.info('Running hourly health check');
 *   return c.text('OK');
 * }));
 * ```
 *
 * @param schedule - Cron expression (e.g., '0 0 * * *' for daily at midnight)
 * @param handler - Handler function to run on schedule
 * @returns Hono handler for cron endpoint
 */
export declare function cron<E extends Env = Env>(schedule: string, handler: CronHandler<E>): Handler<E>;
/**
 * Metadata interface for cron jobs (can be used for registration/discovery).
 */
export interface CronMetadata {
    schedule: string;
    path: string;
}
//# sourceMappingURL=cron.d.ts.map