import { useCallback, useContext, useEffect, useMemo, useRef, useState } from 'react';
import { WebRTCManager, buildUrl, } from '@agentuity/frontend';
import { AgentuityContext } from './context';
/**
 * Map new state to legacy status for backwards compatibility
 */
function stateToStatus(state) {
    if (state === 'idle')
        return 'disconnected';
    if (state === 'negotiating')
        return 'connecting';
    return state;
}
/**
 * React hook for WebRTC peer-to-peer audio/video calls.
 *
 * Handles WebRTC signaling, media capture, and peer connection management.
 *
 * @example
 * ```tsx
 * function VideoCall({ roomId }: { roomId: string }) {
 *   const {
 *     localVideoRef,
 *     remoteVideoRef,
 *     state,
 *     hangup,
 *     muteAudio,
 *     isAudioMuted,
 *   } = useWebRTCCall({
 *     roomId,
 *     signalUrl: '/call/signal',
 *     callbacks: {
 *       onStateChange: (from, to, reason) => {
 *         console.log(`State: ${from} → ${to}`, reason);
 *       },
 *       onConnect: () => console.log('Connected!'),
 *       onDisconnect: (reason) => console.log('Disconnected:', reason),
 *     },
 *   });
 *
 *   return (
 *     <div>
 *       <video ref={localVideoRef} autoPlay muted playsInline />
 *       <video ref={remoteVideoRef} autoPlay playsInline />
 *       <p>State: {state}</p>
 *       <button onClick={() => muteAudio(!isAudioMuted)}>
 *         {isAudioMuted ? 'Unmute' : 'Mute'}
 *       </button>
 *       <button onClick={hangup}>Hang Up</button>
 *     </div>
 *   );
 * }
 * ```
 */
export function useWebRTCCall(options) {
    const context = useContext(AgentuityContext);
    const managerRef = useRef(null);
    const localVideoRef = useRef(null);
    const remoteVideoRef = useRef(null);
    const [state, setState] = useState('idle');
    const [error, setError] = useState(null);
    const [peerId, setPeerId] = useState(null);
    const [remotePeerId, setRemotePeerId] = useState(null);
    const [isAudioMuted, setIsAudioMuted] = useState(false);
    const [isVideoMuted, setIsVideoMuted] = useState(false);
    // Store user callbacks in a ref to avoid recreating manager
    const userCallbacksRef = useRef(options.callbacks);
    userCallbacksRef.current = options.callbacks;
    // Build full signaling URL
    const signalUrl = useMemo(() => {
        // If it's already a full URL, use as-is
        if (options.signalUrl.startsWith('ws://') || options.signalUrl.startsWith('wss://')) {
            return options.signalUrl;
        }
        // Build from context base URL
        const base = context?.baseUrl ?? window.location.origin;
        const wsBase = base.replace(/^http(s?):/, 'ws$1:');
        return buildUrl(wsBase, options.signalUrl);
    }, [context?.baseUrl, options.signalUrl]);
    // Create manager options - use refs to avoid recreating manager on state changes
    const managerOptions = useMemo(() => {
        return {
            signalUrl,
            roomId: options.roomId,
            polite: options.polite,
            iceServers: options.iceServers,
            media: options.media,
            callbacks: {
                onStateChange: (from, to, reason) => {
                    setState(to);
                    if (managerRef.current) {
                        const managerState = managerRef.current.getState();
                        setPeerId(managerState.peerId);
                        setRemotePeerId(managerState.remotePeerId);
                    }
                    userCallbacksRef.current?.onStateChange?.(from, to, reason);
                },
                onConnect: () => {
                    userCallbacksRef.current?.onConnect?.();
                },
                onDisconnect: (reason) => {
                    userCallbacksRef.current?.onDisconnect?.(reason);
                },
                onLocalStream: (stream) => {
                    if (localVideoRef.current) {
                        localVideoRef.current.srcObject = stream;
                    }
                    userCallbacksRef.current?.onLocalStream?.(stream);
                },
                onRemoteStream: (stream) => {
                    if (remoteVideoRef.current) {
                        remoteVideoRef.current.srcObject = stream;
                    }
                    userCallbacksRef.current?.onRemoteStream?.(stream);
                },
                onTrackAdded: (track, stream) => {
                    userCallbacksRef.current?.onTrackAdded?.(track, stream);
                },
                onTrackRemoved: (track) => {
                    userCallbacksRef.current?.onTrackRemoved?.(track);
                },
                onPeerJoined: (id) => {
                    setRemotePeerId(id);
                    userCallbacksRef.current?.onPeerJoined?.(id);
                },
                onPeerLeft: (id) => {
                    setRemotePeerId((current) => (current === id ? null : current));
                    userCallbacksRef.current?.onPeerLeft?.(id);
                },
                onNegotiationStart: () => {
                    userCallbacksRef.current?.onNegotiationStart?.();
                },
                onNegotiationComplete: () => {
                    userCallbacksRef.current?.onNegotiationComplete?.();
                },
                onIceCandidate: (candidate) => {
                    userCallbacksRef.current?.onIceCandidate?.(candidate);
                },
                onIceStateChange: (iceState) => {
                    userCallbacksRef.current?.onIceStateChange?.(iceState);
                },
                onError: (err, currentState) => {
                    setError(err);
                    userCallbacksRef.current?.onError?.(err, currentState);
                },
            },
        };
    }, [signalUrl, options.roomId, options.polite, options.iceServers, options.media]);
    // Initialize manager
    useEffect(() => {
        const manager = new WebRTCManager(managerOptions);
        managerRef.current = manager;
        // Auto-connect if enabled (default: true)
        if (options.autoConnect !== false) {
            manager.connect();
        }
        return () => {
            manager.dispose();
            managerRef.current = null;
        };
    }, [managerOptions, options.autoConnect]);
    const connect = useCallback(() => {
        managerRef.current?.connect();
    }, []);
    const hangup = useCallback(() => {
        managerRef.current?.hangup();
    }, []);
    const muteAudio = useCallback((muted) => {
        managerRef.current?.muteAudio(muted);
        setIsAudioMuted(muted);
    }, []);
    const muteVideo = useCallback((muted) => {
        managerRef.current?.muteVideo(muted);
        setIsVideoMuted(muted);
    }, []);
    return {
        localVideoRef,
        remoteVideoRef,
        state,
        status: stateToStatus(state),
        error,
        peerId,
        remotePeerId,
        isAudioMuted,
        isVideoMuted,
        connect,
        hangup,
        muteAudio,
        muteVideo,
    };
}
//# sourceMappingURL=webrtc.js.map