import { type WebRTCStatus, type WebRTCConnectionState, type WebRTCClientCallbacks } from '@agentuity/frontend';
export type { WebRTCClientCallbacks };
export type { WebRTCStatus, WebRTCConnectionState };
/**
 * Options for useWebRTCCall hook
 */
export interface UseWebRTCCallOptions {
    /** Room ID to join */
    roomId: string;
    /** WebSocket signaling URL (e.g., '/call/signal' or full URL) */
    signalUrl: string;
    /** Whether this peer is "polite" in perfect negotiation (default: true for first joiner) */
    polite?: boolean;
    /** ICE servers configuration */
    iceServers?: RTCIceServer[];
    /** Media constraints for getUserMedia */
    media?: MediaStreamConstraints;
    /** Whether to auto-connect on mount (default: true) */
    autoConnect?: boolean;
    /**
     * Optional callbacks for WebRTC events.
     * These are called in addition to the hook's internal state management.
     */
    callbacks?: Partial<WebRTCClientCallbacks>;
}
/**
 * Return type for useWebRTCCall hook
 */
export interface UseWebRTCCallResult {
    /** Ref to attach to local video element */
    localVideoRef: React.RefObject<HTMLVideoElement | null>;
    /** Ref to attach to remote video element */
    remoteVideoRef: React.RefObject<HTMLVideoElement | null>;
    /** Current connection state (new state machine) */
    state: WebRTCConnectionState;
    /** @deprecated Use `state` instead. Current connection status */
    status: WebRTCStatus;
    /** Current error if any */
    error: Error | null;
    /** Local peer ID assigned by server */
    peerId: string | null;
    /** Remote peer ID */
    remotePeerId: string | null;
    /** Whether audio is muted */
    isAudioMuted: boolean;
    /** Whether video is muted */
    isVideoMuted: boolean;
    /** Manually start the connection (if autoConnect is false) */
    connect: () => void;
    /** End the call */
    hangup: () => void;
    /** Mute or unmute audio */
    muteAudio: (muted: boolean) => void;
    /** Mute or unmute video */
    muteVideo: (muted: boolean) => void;
}
/**
 * React hook for WebRTC peer-to-peer audio/video calls.
 *
 * Handles WebRTC signaling, media capture, and peer connection management.
 *
 * @example
 * ```tsx
 * function VideoCall({ roomId }: { roomId: string }) {
 *   const {
 *     localVideoRef,
 *     remoteVideoRef,
 *     state,
 *     hangup,
 *     muteAudio,
 *     isAudioMuted,
 *   } = useWebRTCCall({
 *     roomId,
 *     signalUrl: '/call/signal',
 *     callbacks: {
 *       onStateChange: (from, to, reason) => {
 *         console.log(`State: ${from} → ${to}`, reason);
 *       },
 *       onConnect: () => console.log('Connected!'),
 *       onDisconnect: (reason) => console.log('Disconnected:', reason),
 *     },
 *   });
 *
 *   return (
 *     <div>
 *       <video ref={localVideoRef} autoPlay muted playsInline />
 *       <video ref={remoteVideoRef} autoPlay playsInline />
 *       <p>State: {state}</p>
 *       <button onClick={() => muteAudio(!isAudioMuted)}>
 *         {isAudioMuted ? 'Unmute' : 'Mute'}
 *       </button>
 *       <button onClick={hangup}>Hang Up</button>
 *     </div>
 *   );
 * }
 * ```
 */
export declare function useWebRTCCall(options: UseWebRTCCallOptions): UseWebRTCCallResult;
//# sourceMappingURL=webrtc.d.ts.map