import type { WebRTCConnectionState, WebRTCDisconnectReason } from '@agentuity/core';
/**
 * Callbacks for WebRTC client state changes and events.
 * All callbacks are optional - only subscribe to events you care about.
 */
export interface WebRTCClientCallbacks {
    /**
     * Called on every state transition.
     * @param from - Previous state
     * @param to - New state
     * @param reason - Optional reason for the transition
     */
    onStateChange?: (from: WebRTCConnectionState, to: WebRTCConnectionState, reason?: string) => void;
    /**
     * Called when connection is fully established.
     */
    onConnect?: () => void;
    /**
     * Called when disconnected from the call.
     * @param reason - Why the disconnection happened
     */
    onDisconnect?: (reason: WebRTCDisconnectReason) => void;
    /**
     * Called when local media stream is acquired.
     * @param stream - The local MediaStream
     */
    onLocalStream?: (stream: MediaStream) => void;
    /**
     * Called when remote media stream is received.
     * @param stream - The remote MediaStream
     */
    onRemoteStream?: (stream: MediaStream) => void;
    /**
     * Called when a new track is added to a stream.
     * @param track - The added track
     * @param stream - The stream containing the track
     */
    onTrackAdded?: (track: MediaStreamTrack, stream: MediaStream) => void;
    /**
     * Called when a track is removed from a stream.
     * @param track - The removed track
     */
    onTrackRemoved?: (track: MediaStreamTrack) => void;
    /**
     * Called when a peer joins the room.
     * @param peerId - The peer's ID
     */
    onPeerJoined?: (peerId: string) => void;
    /**
     * Called when a peer leaves the room.
     * @param peerId - The peer's ID
     */
    onPeerLeft?: (peerId: string) => void;
    /**
     * Called when SDP/ICE negotiation starts.
     */
    onNegotiationStart?: () => void;
    /**
     * Called when SDP/ICE negotiation completes successfully.
     */
    onNegotiationComplete?: () => void;
    /**
     * Called for each ICE candidate generated.
     * @param candidate - The ICE candidate
     */
    onIceCandidate?: (candidate: RTCIceCandidateInit) => void;
    /**
     * Called when ICE connection state changes.
     * @param state - The new ICE connection state
     */
    onIceStateChange?: (state: string) => void;
    /**
     * Called when an error occurs.
     * @param error - The error that occurred
     * @param state - The state when the error occurred
     */
    onError?: (error: Error, state: WebRTCConnectionState) => void;
}
/**
 * @deprecated Use `WebRTCConnectionState` from @agentuity/core instead.
 */
export type WebRTCStatus = 'disconnected' | 'connecting' | 'signaling' | 'connected';
/**
 * @deprecated Use `WebRTCClientCallbacks` from @agentuity/core instead.
 */
export interface WebRTCCallbacks {
    onLocalStream?: (stream: MediaStream) => void;
    onRemoteStream?: (stream: MediaStream) => void;
    onStatusChange?: (status: WebRTCStatus) => void;
    onError?: (error: Error) => void;
    onPeerJoined?: (peerId: string) => void;
    onPeerLeft?: (peerId: string) => void;
}
/**
 * Options for WebRTCManager
 */
export interface WebRTCManagerOptions {
    /** WebSocket signaling URL */
    signalUrl: string;
    /** Room ID to join */
    roomId: string;
    /** Whether this peer is "polite" in perfect negotiation (default: true) */
    polite?: boolean;
    /** ICE servers configuration */
    iceServers?: RTCIceServer[];
    /** Media constraints for getUserMedia */
    media?: MediaStreamConstraints;
    /**
     * Callbacks for state changes and events.
     * Supports both legacy WebRTCCallbacks and new WebRTCClientCallbacks.
     */
    callbacks?: WebRTCClientCallbacks;
}
/**
 * WebRTC manager state
 */
export interface WebRTCManagerState {
    state: WebRTCConnectionState;
    peerId: string | null;
    remotePeerId: string | null;
    isAudioMuted: boolean;
    isVideoMuted: boolean;
    /** @deprecated Use `state` instead */
    status: WebRTCStatus;
}
/**
 * Framework-agnostic WebRTC connection manager with signaling,
 * perfect negotiation, and media stream handling.
 *
 * Uses an explicit state machine for connection lifecycle:
 * - idle: No resources allocated, ready to connect
 * - connecting: Acquiring media + opening WebSocket
 * - signaling: In room, waiting for peer
 * - negotiating: SDP/ICE exchange in progress
 * - connected: Media flowing
 *
 * @example
 * ```ts
 * const manager = new WebRTCManager({
 *   signalUrl: 'wss://example.com/call/signal',
 *   roomId: 'my-room',
 *   callbacks: {
 *     onStateChange: (from, to, reason) => {
 *       console.log(`State: ${from} → ${to}`, reason);
 *     },
 *     onConnect: () => console.log('Connected!'),
 *     onDisconnect: (reason) => console.log('Disconnected:', reason),
 *     onLocalStream: (stream) => { localVideo.srcObject = stream; },
 *     onRemoteStream: (stream) => { remoteVideo.srcObject = stream; },
 *     onError: (error, state) => console.error(`Error in ${state}:`, error),
 *   },
 * });
 *
 * await manager.connect();
 * ```
 */
export declare class WebRTCManager {
    private ws;
    private pc;
    private localStream;
    private remoteStream;
    private peerId;
    private remotePeerId;
    private isAudioMuted;
    private isVideoMuted;
    private _state;
    private makingOffer;
    private ignoreOffer;
    private polite;
    private pendingCandidates;
    private hasRemoteDescription;
    private options;
    private callbacks;
    constructor(options: WebRTCManagerOptions);
    /**
     * Current connection state
     */
    get state(): WebRTCConnectionState;
    /**
     * Get current manager state
     */
    getState(): WebRTCManagerState;
    /**
     * Get local media stream
     */
    getLocalStream(): MediaStream | null;
    /**
     * Get remote media stream
     */
    getRemoteStream(): MediaStream | null;
    /**
     * Transition to a new state with callback notifications
     */
    private setState;
    private mapToDisconnectReason;
    private send;
    /**
     * Connect to the signaling server and start the call
     */
    connect(): Promise<void>;
    private handleSignalingMessage;
    private createPeerConnection;
    private createOffer;
    private handleRemoteSDP;
    private handleRemoteICE;
    private closePeerConnection;
    /**
     * End the call and disconnect
     */
    hangup(): void;
    /**
     * Mute or unmute audio
     */
    muteAudio(muted: boolean): void;
    /**
     * Mute or unmute video
     */
    muteVideo(muted: boolean): void;
    /**
     * Clean up all resources
     */
    dispose(): void;
}
//# sourceMappingURL=webrtc-manager.d.ts.map