import type { Logger } from '@agentuity/core';
import { DeploymentConfig } from '@agentuity/server';
import type * as z from 'zod';
import { z as zod } from 'zod';
import type { APIClient } from './api';
export { DeploymentConfig };
export type { Logger };
export declare const ConfigSchema: z.ZodObject<{
    name: z.ZodString;
    auth: z.ZodOptional<z.ZodObject<{
        api_key: z.ZodOptional<z.ZodString>;
        user_id: z.ZodOptional<z.ZodString>;
        expires: z.ZodOptional<z.ZodNumber>;
    }, z.core.$strip>>;
    devmode: z.ZodOptional<z.ZodObject<{
        hostname: z.ZodOptional<z.ZodString>;
    }, z.core.$strip>>;
    overrides: z.ZodNullable<z.ZodOptional<z.ZodObject<{
        api_url: z.ZodOptional<z.ZodURL>;
        app_url: z.ZodOptional<z.ZodURL>;
        transport_url: z.ZodOptional<z.ZodURL>;
        stream_url: z.ZodOptional<z.ZodURL>;
        kv_url: z.ZodOptional<z.ZodURL>;
        sandbox_url: z.ZodOptional<z.ZodURL>;
        vector_url: z.ZodOptional<z.ZodURL>;
        catalyst_url: z.ZodOptional<z.ZodURL>;
        ion_url: z.ZodOptional<z.ZodURL>;
        gravity_url: z.ZodOptional<z.ZodURL>;
        skip_version_check: z.ZodOptional<z.ZodBoolean>;
    }, z.core.$strip>>>;
    preferences: z.ZodOptional<z.ZodObject<{
        last_update_check: z.ZodOptional<z.ZodNumber>;
        last_legacy_warning: z.ZodOptional<z.ZodNumber>;
        signup_banner_shown: z.ZodOptional<z.ZodBoolean>;
        orgId: z.ZodOptional<z.ZodString>;
        region: z.ZodOptional<z.ZodString>;
        project_dir: z.ZodOptional<z.ZodString>;
    }, z.core.$strip>>;
    gravity: z.ZodOptional<z.ZodObject<{
        version: z.ZodOptional<z.ZodString>;
        checked: z.ZodOptional<z.ZodNumber>;
    }, z.core.$strip>>;
}, z.core.$strip>;
export type Config = zod.infer<typeof ConfigSchema>;
export type LogLevel = 'debug' | 'trace' | 'info' | 'warn' | 'error';
/**
 * Build phases for the bundler
 */
export type BuildPhase = 'api' | 'web' | 'workbench';
/**
 * Context provided to the build config function
 */
export interface BuildContext {
    /**
     * The root directory of the project
     */
    rootDir: string;
    /**
     * Whether this is a development build
     */
    dev: boolean;
    /**
     * The output directory for the build
     */
    outDir: string;
    /**
     * The source directory
     */
    srcDir: string;
    /**
     * Organization ID (if available)
     */
    orgId?: string;
    /**
     * Project ID (if available)
     */
    projectId?: string;
    /**
     * Deployment region
     */
    region: string;
    /**
     * Logger instance
     */
    logger: Logger;
}
/**
 * Workbench configuration
 *
 * Presence of this config object implicitly enables workbench in dev mode.
 * To disable workbench, omit this config entirely.
 */
export interface WorkbenchConfig {
    /**
     * Route where the workbench UI will be served
     * @default '/workbench'
     */
    route?: string;
    /**
     * Custom headers to send with workbench requests
     */
    headers?: Record<string, string>;
}
/**
 * Web analytics configuration for SDK-created applications
 */
export interface AnalyticsConfig {
    /**
     * Enable/disable analytics
     * @default true
     */
    enabled?: boolean;
    /**
     * Require explicit user consent before tracking
     * When true, analytics is no-op until optIn() is called
     * @default false
     */
    requireConsent?: boolean;
    /**
     * Track click events on elements with data-analytics attribute
     * @default true
     */
    trackClicks?: boolean;
    /**
     * Track scroll depth (25%, 50%, 75%, 100%)
     * @default true
     */
    trackScroll?: boolean;
    /**
     * Track outbound link clicks
     * @default true
     */
    trackOutboundLinks?: boolean;
    /**
     * Track form submissions
     * @default false
     */
    trackForms?: boolean;
    /**
     * Track Core Web Vitals
     * @default true
     */
    trackWebVitals?: boolean;
    /**
     * Track JavaScript errors
     * @default true
     */
    trackErrors?: boolean;
    /**
     * Track SPA navigation (popstate, pushState)
     * Automatically tracks virtual pageviews on client-side route changes
     * @default true
     */
    trackSPANavigation?: boolean;
    /**
     * Sample rate (0-1). 1 = 100% of events
     * @default 1
     */
    sampleRate?: number;
    /**
     * URL patterns to exclude from tracking (regex strings)
     */
    excludePatterns?: string[];
    /**
     * Custom data to include with every event
     */
    globalProperties?: Record<string, unknown>;
}
/**
 * Agentuity project configuration (declarative)
 */
export interface AgentuityConfig {
    /**
     * Workbench configuration
     */
    workbench?: WorkbenchConfig;
    /**
     * Web analytics configuration
     * Set to false to disable, or provide options object
     * @default true (enabled with defaults)
     */
    analytics?: boolean | AnalyticsConfig;
    /**
     * Vite plugins to add to the client build
     * These are added AFTER Agentuity's built-in plugins
     */
    plugins?: Array<import('vite').Plugin>;
    /**
     * Additional define constants for code replacement in Vite builds
     * These are merged with Agentuity's default defines
     * Note: Cannot override AGENTUITY_PUBLIC_* or process.env.NODE_ENV
     */
    define?: Record<string, string>;
}
/**
 * User-provided build configuration for a specific phase (legacy Bun bundler)
 * @deprecated Use AgentuityConfig instead
 */
export interface BuildConfig {
    /**
     * Additional Bun plugins to apply during bundling
     * These are added AFTER the Agentuity plugin
     */
    plugins?: Array<import('bun').BunPlugin>;
    /**
     * Additional external modules to exclude from bundling
     * These are merged with Agentuity's default externals
     */
    external?: string[];
    /**
     * Additional define constants for code replacement
     * These are merged with Agentuity's default defines
     * Note: Cannot override process.env.AGENTUITY_* or process.env.NODE_ENV
     */
    define?: Record<string, string>;
}
/**
 * Configuration function that users export from agentuity.config.ts
 */
export type BuildConfigFunction = (phase: BuildPhase, context: BuildContext) => BuildConfig | Promise<BuildConfig>;
export interface Profile {
    name: string;
    filename: string;
    selected: boolean;
}
export interface AuthData {
    apiKey: string;
    userId: string;
    expires: Date;
}
export interface GlobalOptions {
    config?: string;
    logLevel: LogLevel;
    logTimestamp?: boolean;
    logPrefix?: boolean;
    orgId?: string;
    errorFormat?: 'json' | 'text';
    json?: boolean;
    quiet?: boolean;
    noProgress?: boolean;
    color?: 'auto' | 'always' | 'never';
    explain?: boolean;
    dryRun?: boolean;
    validate?: boolean;
    skipVersionCheck?: boolean;
}
export interface PaginationInfo {
    supported: boolean;
    defaultLimit?: number;
    maxLimit?: number;
    parameters?: {
        limit?: string;
        offset?: string;
        cursor?: string;
    };
}
export interface CommandExample {
    command: string;
    description: string;
}
export interface CommandSchemas {
    args?: z.ZodType;
    options?: z.ZodType;
    response?: z.ZodType;
    aliases?: Record<string, string[]>;
}
export type ProjectConfig = zod.infer<typeof ProjectSchema>;
export type Requires = {
    auth?: true;
    project?: true;
    apiClient?: true;
    org?: true;
    regions?: true;
    region?: true;
};
export type Optional = {
    auth?: true | string;
    project?: true;
    org?: true;
    region?: true;
};
type AddArgs<A extends z.ZodType | undefined> = A extends z.ZodType ? {
    args: z.infer<A>;
} : Record<string, never>;
type AddOpts<O extends z.ZodType | undefined> = O extends z.ZodType ? {
    opts: z.infer<O>;
} : Record<string, never>;
type AuthMode<R extends Requires | undefined, O extends Optional | undefined> = R extends {
    auth: true;
} ? 'required' : O extends {
    auth: true | string;
} ? 'optional' : 'none';
type ProjectMode<R extends Requires | undefined, O extends Optional | undefined> = R extends {
    project: true;
} ? 'required' : O extends {
    project: true;
} ? 'optional' : 'none';
type APIClientRequired<R extends Requires | undefined> = R extends {
    apiClient: true;
} ? true : false;
type AddAuth<M extends 'required' | 'optional' | 'none'> = M extends 'required' ? {
    auth: AuthData;
} : M extends 'optional' ? {
    auth: AuthData | undefined;
} : Record<string, never>;
type AddProject<M extends 'required' | 'optional' | 'none'> = M extends 'required' ? {
    project: ProjectConfig;
    projectDir: string;
} : M extends 'optional' ? {
    project?: ProjectConfig;
    projectDir: string;
} : Record<string, never>;
type OrgMode<R extends Requires | undefined, O extends Optional | undefined> = R extends {
    org: true;
} ? 'required' : O extends {
    org: true;
} ? 'optional' : 'none';
type AddOrg<M extends 'required' | 'optional' | 'none'> = M extends 'required' ? {
    orgId: string;
} : M extends 'optional' ? {
    orgId?: string;
} : Record<string, never>;
type RegionsRequired<R extends Requires | undefined> = R extends {
    regions: true;
} ? true : false;
type AddRegions<R extends boolean> = R extends true ? {
    regions: Array<{
        region: string;
        description: string;
    }>;
} : Record<string, never>;
type RegionMode<R extends Requires | undefined, O extends Optional | undefined> = R extends {
    region: true;
} ? 'required' : O extends {
    region: true;
} ? 'optional' : 'none';
type AddRegionOptional<M extends 'required' | 'optional' | 'none'> = M extends 'required' ? {
    region: string;
} : M extends 'optional' ? {
    region?: string;
} : Record<string, never>;
export type CommandContextFromSpecs<R extends Requires | undefined, O extends Optional | undefined, A extends z.ZodType | undefined = undefined, Op extends z.ZodType | undefined = undefined> = {
    config: Config | null;
    logger: Logger;
    options: GlobalOptions;
} & AddArgs<A> & AddOpts<Op> & AddAuth<AuthMode<R, O>> & AddProject<ProjectMode<R, O>> & AddOrg<OrgMode<R, O>> & AddRegions<RegionsRequired<R>> & AddRegionOptional<RegionMode<R, O>> & (APIClientRequired<R> extends true ? {
    apiClient: APIClient;
} : Record<string, never>);
export type CommandContext<R extends Requires | undefined = undefined, O extends Optional | undefined = undefined, A extends z.ZodType | undefined = undefined, Op extends z.ZodType | undefined = undefined> = CommandContextFromSpecs<R, O, A, Op>;
export type WebUrl<R extends Requires | undefined = undefined, O extends Optional | undefined = undefined, A extends z.ZodType | undefined = undefined, Op extends z.ZodType | undefined = undefined> = string | ((ctx: CommandContext<R, O, A, Op>) => string | undefined | null);
export declare function createSubcommand<R extends Requires | undefined = undefined, O extends Optional | undefined = undefined, A extends z.ZodType | undefined = undefined, Op extends z.ZodType | undefined = undefined, Res extends z.ZodType | undefined = undefined>(definition: {
    name: string;
    description: string;
    banner?: true;
    aliases?: string[];
    toplevel?: boolean;
    requires?: R;
    optional?: O;
    examples?: CommandExample[];
    idempotent?: boolean;
    prerequisites?: string[];
    pagination?: PaginationInfo;
    tags?: string[];
    skipSkill?: boolean;
    webUrl?: WebUrl<R, O, A, Op>;
    schema?: A extends z.ZodType ? Op extends z.ZodType ? Res extends z.ZodType ? {
        args: A;
        options: Op;
        response: Res;
        aliases?: Record<string, string[]>;
    } : {
        args: A;
        options: Op;
        response?: z.ZodType;
        aliases?: Record<string, string[]>;
    } : Res extends z.ZodType ? {
        args: A;
        response: Res;
        aliases?: Record<string, string[]>;
    } : {
        args: A;
        response?: z.ZodType;
        aliases?: Record<string, string[]>;
    } : Op extends z.ZodType ? Res extends z.ZodType ? {
        options: Op;
        response: Res;
        aliases?: Record<string, string[]>;
    } : {
        options: Op;
        response?: z.ZodType;
        aliases?: Record<string, string[]>;
    } : Res extends z.ZodType ? {
        response: Res;
        aliases?: Record<string, string[]>;
    } : {
        response?: z.ZodType;
        aliases?: Record<string, string[]>;
    };
    handler(ctx: CommandContext<R, O, A, Op>): Res extends z.ZodType ? z.infer<Res> | Promise<z.infer<Res>> : unknown | Promise<unknown>;
}): SubcommandDefinition;
export declare function createCommand<R extends Requires | undefined = undefined, O extends Optional | undefined = undefined, A extends z.ZodType | undefined = undefined, Op extends z.ZodType | undefined = undefined, Res extends z.ZodType | undefined = undefined>(definition: {
    name: string;
    description: string;
    banner?: true;
    aliases?: string[];
    hidden?: boolean;
    executable?: boolean;
    skipUpgradeCheck?: boolean;
    passThroughArgs?: boolean;
    requires?: R;
    optional?: O;
    examples?: CommandExample[];
    idempotent?: boolean;
    prerequisites?: string[];
    pagination?: PaginationInfo;
    tags?: string[];
    skipSkill?: boolean;
    webUrl?: WebUrl<R, O, A, Op>;
    schema?: A extends z.ZodType ? Op extends z.ZodType ? Res extends z.ZodType ? {
        args: A;
        options: Op;
        response: Res;
        aliases?: Record<string, string[]>;
    } : {
        args: A;
        options: Op;
        response?: z.ZodType;
        aliases?: Record<string, string[]>;
    } : Res extends z.ZodType ? {
        args: A;
        response: Res;
        aliases?: Record<string, string[]>;
    } : {
        args: A;
        response?: z.ZodType;
        aliases?: Record<string, string[]>;
    } : Op extends z.ZodType ? Res extends z.ZodType ? {
        options: Op;
        response: Res;
        aliases?: Record<string, string[]>;
    } : {
        options: Op;
        response?: z.ZodType;
        aliases?: Record<string, string[]>;
    } : Res extends z.ZodType ? {
        response: Res;
        aliases?: Record<string, string[]>;
    } : {
        response?: z.ZodType;
        aliases?: Record<string, string[]>;
    };
    handler?(ctx: CommandContext<R, O, A, Op>): Res extends z.ZodType ? z.infer<Res> | Promise<z.infer<Res>> : unknown | Promise<unknown>;
    subcommands?: SubcommandDefinition[];
}): CommandDefinition;
type CommandDefBase = {
    name: string;
    description: string;
    aliases?: string[];
    banner?: boolean;
    executable?: boolean;
    skipUpgradeCheck?: boolean;
    passThroughArgs?: boolean;
    skipSkill?: boolean;
    examples?: CommandExample[];
    idempotent?: boolean;
    prerequisites?: string[];
    pagination?: PaginationInfo;
    tags?: string[];
    schema?: CommandSchemas;
    webUrl?: string | ((ctx: CommandContext) => string | undefined | null);
    handler(ctx: CommandContext): unknown | Promise<unknown>;
    subcommands?: SubcommandDefinition[];
} | {
    name: string;
    description: string;
    aliases?: string[];
    banner?: boolean;
    executable?: boolean;
    skipUpgradeCheck?: boolean;
    passThroughArgs?: boolean;
    skipSkill?: boolean;
    examples?: CommandExample[];
    idempotent?: boolean;
    prerequisites?: string[];
    pagination?: PaginationInfo;
    tags?: string[];
    schema?: CommandSchemas;
    webUrl?: string | ((ctx: CommandContext) => string | undefined | null);
    handler?: undefined;
    subcommands: SubcommandDefinition[];
};
type SubcommandDefBase = {
    name: string;
    description: string;
    aliases?: string[];
    toplevel?: boolean;
    banner?: boolean;
    skipSkill?: boolean;
    examples?: CommandExample[];
    idempotent?: boolean;
    prerequisites?: string[];
    pagination?: PaginationInfo;
    tags?: string[];
    schema?: CommandSchemas;
    webUrl?: string | ((ctx: CommandContext) => string | undefined | null);
    handler(ctx: CommandContext): unknown | Promise<unknown>;
    subcommands?: SubcommandDefinition[];
} | {
    name: string;
    description: string;
    aliases?: string[];
    toplevel?: boolean;
    banner?: boolean;
    skipSkill?: boolean;
    examples?: CommandExample[];
    idempotent?: boolean;
    prerequisites?: string[];
    pagination?: PaginationInfo;
    tags?: string[];
    schema?: CommandSchemas;
    webUrl?: string | ((ctx: CommandContext) => string | undefined | null);
    handler?: undefined;
    subcommands: SubcommandDefinition[];
};
export type CommandDefinition = (CommandDefBase & {
    hidden?: boolean;
    requires: Requires & {
        auth: true;
    };
    optional?: Optional & {
        auth?: never;
    };
}) | (CommandDefBase & {
    hidden?: boolean;
    requires?: Requires & {
        auth?: never;
    };
    optional: Optional & {
        auth: true | string;
    };
}) | (CommandDefBase & {
    hidden?: boolean;
    requires?: Requires & {
        auth?: never;
    };
    optional?: Optional & {
        auth?: never;
    };
});
export type SubcommandDefinition = (SubcommandDefBase & {
    requires: Requires & {
        auth: true;
    };
    optional?: Optional & {
        auth?: never;
    };
}) | (SubcommandDefBase & {
    requires?: Requires & {
        auth?: never;
    };
    optional: Optional & {
        auth: true | string;
    };
}) | (SubcommandDefBase & {
    requires?: Requires & {
        auth?: never;
    };
    optional?: Optional & {
        auth?: never;
    };
});
export declare const ProjectSchema: z.ZodObject<{
    projectId: z.ZodString;
    orgId: z.ZodString;
    region: z.ZodString;
    deployment: z.ZodOptional<z.ZodObject<{
        resources: z.ZodOptional<z.ZodObject<{
            memory: z.ZodDefault<z.ZodString>;
            cpu: z.ZodDefault<z.ZodString>;
            disk: z.ZodDefault<z.ZodString>;
        }, z.core.$strip>>;
        mode: z.ZodOptional<z.ZodObject<{
            type: z.ZodDefault<z.ZodEnum<{
                "on-demand": "on-demand";
                provisioned: "provisioned";
            }>>;
            idle: z.ZodOptional<z.ZodString>;
        }, z.core.$strip>>;
        dependencies: z.ZodOptional<z.ZodArray<z.ZodString>>;
        domains: z.ZodOptional<z.ZodArray<z.ZodString>>;
    }, z.core.$strip>>;
    skipGitSetup: z.ZodOptional<z.ZodBoolean>;
}, z.core.$strip>;
export declare const BuildMetadataSchema: z.ZodObject<{
    routes: z.ZodArray<z.ZodObject<{
        id: z.ZodString;
        filename: z.ZodString;
        path: z.ZodString;
        method: z.ZodEnum<{
            get: "get";
            post: "post";
            put: "put";
            delete: "delete";
            patch: "patch";
        }>;
        version: z.ZodString;
        type: z.ZodEnum<{
            api: "api";
            sms: "sms";
            email: "email";
            cron: "cron";
            websocket: "websocket";
            sse: "sse";
            stream: "stream";
        }>;
        agentIds: z.ZodOptional<z.ZodArray<z.ZodString>>;
        config: z.ZodOptional<z.ZodRecord<z.ZodString, z.ZodUnknown>>;
        schema: z.ZodOptional<z.ZodObject<{
            input: z.ZodOptional<z.ZodString>;
            output: z.ZodOptional<z.ZodString>;
        }, z.core.$strip>>;
    }, z.core.$strip>>;
    agents: z.ZodArray<z.ZodObject<{
        id: z.ZodString;
        agentId: z.ZodString;
        projectId: z.ZodString;
        name: z.ZodString;
        description: z.ZodOptional<z.ZodString>;
        evals: z.ZodOptional<z.ZodArray<z.ZodObject<{
            id: z.ZodString;
            identifier: z.ZodString;
            name: z.ZodString;
            description: z.ZodOptional<z.ZodString>;
            agentIdentifier: z.ZodString;
            projectId: z.ZodString;
            filename: z.ZodString;
            version: z.ZodString;
        }, z.core.$strip>>>;
        schema: z.ZodOptional<z.ZodObject<{
            input: z.ZodOptional<z.ZodString>;
            output: z.ZodOptional<z.ZodString>;
        }, z.core.$strip>>;
        filename: z.ZodString;
        version: z.ZodString;
    }, z.core.$strip>>;
    assets: z.ZodArray<z.ZodObject<{
        filename: z.ZodString;
        kind: z.ZodString;
        contentType: z.ZodString;
        contentEncoding: z.ZodOptional<z.ZodString>;
        size: z.ZodNumber;
    }, z.core.$strip>>;
    project: z.ZodObject<{
        id: z.ZodString;
        name: z.ZodString;
        version: z.ZodOptional<z.ZodString>;
        description: z.ZodOptional<z.ZodString>;
        keywords: z.ZodOptional<z.ZodArray<z.ZodString>>;
        orgId: z.ZodString;
    }, z.core.$strip>;
    deployment: z.ZodIntersection<z.ZodObject<{
        resources: z.ZodOptional<z.ZodObject<{
            memory: z.ZodDefault<z.ZodString>;
            cpu: z.ZodDefault<z.ZodString>;
            disk: z.ZodDefault<z.ZodString>;
        }, z.core.$strip>>;
        mode: z.ZodOptional<z.ZodObject<{
            type: z.ZodDefault<z.ZodEnum<{
                "on-demand": "on-demand";
                provisioned: "provisioned";
            }>>;
            idle: z.ZodOptional<z.ZodString>;
        }, z.core.$strip>>;
        dependencies: z.ZodOptional<z.ZodArray<z.ZodString>>;
        domains: z.ZodOptional<z.ZodArray<z.ZodString>>;
    }, z.core.$strip>, z.ZodObject<{
        id: z.ZodString;
        date: z.ZodString;
        git: z.ZodOptional<z.ZodObject<{
            repo: z.ZodOptional<z.ZodString>;
            commit: z.ZodOptional<z.ZodString>;
            message: z.ZodOptional<z.ZodString>;
            branch: z.ZodOptional<z.ZodString>;
            tags: z.ZodOptional<z.ZodArray<z.ZodString>>;
            pr: z.ZodOptional<z.ZodString>;
            provider: z.ZodOptional<z.ZodString>;
            trigger: z.ZodOptional<z.ZodDefault<z.ZodString>>;
            url: z.ZodOptional<z.ZodString>;
            buildUrl: z.ZodOptional<z.ZodString>;
            event: z.ZodOptional<z.ZodDefault<z.ZodEnum<{
                push: "push";
                pull_request: "pull_request";
                manual: "manual";
                workflow: "workflow";
            }>>>;
            pull_request: z.ZodOptional<z.ZodObject<{
                number: z.ZodNumber;
                url: z.ZodOptional<z.ZodString>;
            }, z.core.$strip>>;
        }, z.core.$strip>>;
        build: z.ZodObject<{
            bun: z.ZodString;
            agentuity: z.ZodString;
            arch: z.ZodString;
            platform: z.ZodString;
        }, z.core.$strip>;
    }, z.core.$strip>>;
}, z.core.$strip>;
export type BuildMetadata = zod.infer<typeof BuildMetadataSchema>;
export type Project = zod.infer<typeof ProjectSchema>;
export declare const DeployOptionsSchema: z.ZodObject<{
    logsUrl: z.ZodOptional<z.ZodURL>;
    trigger: z.ZodOptional<z.ZodDefault<z.ZodEnum<{
        cli: "cli";
        workflow: "workflow";
        webhook: "webhook";
    }>>>;
    commitUrl: z.ZodOptional<z.ZodURL>;
    message: z.ZodOptional<z.ZodString>;
    commit: z.ZodOptional<z.ZodString>;
    branch: z.ZodOptional<z.ZodString>;
    provider: z.ZodOptional<z.ZodString>;
    repo: z.ZodOptional<z.ZodString>;
    event: z.ZodOptional<z.ZodDefault<z.ZodEnum<{
        workflow: "workflow";
        pull_request: "pull_request";
        push: "push";
        manual: "manual";
    }>>>;
    pullRequestNumber: z.ZodOptional<z.ZodNumber>;
    pullRequestUrl: z.ZodOptional<z.ZodURL>;
}, z.core.$strip>;
export type DeployOptions = z.infer<typeof DeployOptionsSchema>;
//# sourceMappingURL=types.d.ts.map