/**
 * Reusable REPL (Read-Eval-Print Loop) component for building interactive CLI tools
 */
import * as tui from './tui';
import { z } from 'zod';
/**
 * Result of parsing a REPL command
 */
export interface ParsedCommand {
    /** The command name */
    command: string;
    /** Command arguments (positional) */
    args: string[];
    /** Command options (flags/named parameters) */
    options: Record<string, string | boolean>;
}
/**
 * Table column definition
 */
export type TableColumn = tui.TableColumn;
/**
 * Context provided to command handlers
 */
export interface ReplContext {
    /** The parsed command */
    parsed: ParsedCommand;
    /** Raw input line */
    raw: string;
    /** Write output to the REPL */
    write: (message: string) => void;
    /** Write error output to the REPL */
    error: (message: string) => void;
    /** Write success output to the REPL */
    success: (message: string) => void;
    /** Write info output to the REPL */
    info: (message: string) => void;
    /** Write warning output to the REPL */
    warning: (message: string) => void;
    /** Write debug output to the REPL */
    debug: (message: string) => void;
    /** Update the progress/activity message */
    setProgress: (message: string) => void;
    /** Abort signal for cancelling long-running operations */
    signal: AbortSignal;
    /** Exit the REPL */
    exit: () => void;
    /** Render a table with columns and data */
    table: (columns: TableColumn[], data: Record<string, unknown>[]) => void;
    /** Render colorized JSON output */
    json: (value: unknown) => void;
}
/**
 * Command handler function - can return void, Promise<void>, or an async generator for streaming
 */
export type CommandHandler = (ctx: ReplContext) => void | Promise<void> | AsyncGenerator<string, void, unknown>;
/**
 * Command schema for validation and autocomplete
 */
export interface ReplCommandSchema {
    /** Zod schema for arguments (positional) */
    args?: z.ZodTuple<[z.ZodTypeAny, ...z.ZodTypeAny[]]> | z.ZodArray<z.ZodTypeAny>;
    /** Zod schema for options (flags) */
    options?: z.ZodObject<z.ZodRawShape>;
    /** Argument names for display (e.g., ['message', 'count']) */
    argNames?: string[];
}
/**
 * Subcommand definition
 */
export interface ReplSubcommand {
    /** Subcommand name */
    name: string;
    /** Subcommand description */
    description?: string;
    /** Subcommand handler */
    handler: CommandHandler;
    /** Aliases for this subcommand */
    aliases?: string[];
    /** Schema for validation and autocomplete hints */
    schema?: ReplCommandSchema;
}
/**
 * Command definition for the REPL
 */
export interface ReplCommand {
    /** Command name */
    name: string;
    /** Command description (shown in help) */
    description?: string;
    /** Command handler (not used if subcommands provided) */
    handler?: CommandHandler;
    /** Aliases for this command */
    aliases?: string[];
    /** Schema for validation and autocomplete hints */
    schema?: ReplCommandSchema;
    /** Subcommands for this command group */
    subcommands?: ReplSubcommand[];
}
/**
 * REPL configuration
 */
export interface ReplConfig {
    /** REPL prompt (default: "> ") */
    prompt?: string;
    /** Welcome message shown on startup */
    welcome?: string;
    /** Exit message shown on exit */
    exitMessage?: string;
    /** Commands to register */
    commands: ReplCommand[];
    /** Show help command automatically (default: true) */
    showHelp?: boolean;
    /** Name for history file (saved as ~/.config/agentuity/history/<name>.txt) */
    name?: string;
}
/**
 * Create and run a REPL
 */
export declare function createRepl(config: ReplConfig): Promise<void>;
//# sourceMappingURL=repl.d.ts.map