import { ServiceException, toServiceException, fromResponse } from '@agentuity/core';
const sensitiveHeaders = new Set(['authorization', 'x-api-key']);
/**
 * Redacts the middle of a string while keeping a prefix and suffix visible.
 * Ensures that if the string is too short, everything is redacted.
 *
 * @param input The string to redact
 * @param prefix Number of chars to keep at the start
 * @param suffix Number of chars to keep at the end
 * @param mask  Character used for redaction
 */
export function redact(input, prefix = 4, suffix = 4, mask = '*') {
    if (!input)
        return '';
    // If revealing prefix+suffix would leak too much, fully mask
    if (input.length <= prefix + suffix) {
        return mask.repeat(input.length);
    }
    const start = input.slice(0, prefix);
    const end = input.slice(-suffix);
    const hiddenLength = input.length - prefix - suffix;
    return start + mask.repeat(hiddenLength) + end;
}
const redactHeaders = (kv) => {
    const values = [];
    for (const k of Object.keys(kv)) {
        const _k = k.toLowerCase();
        const v = kv[k];
        if (sensitiveHeaders.has(_k)) {
            if (_k === 'authorization' && v.startsWith('Bearer ')) {
                values.push(`${_k}=Bearer ${redact(v.substring(7))}`);
            }
            else {
                values.push(`${_k}=${redact(v)}`);
            }
        }
        else {
            values.push(`${_k}=${v}`);
        }
    }
    return '[' + values.join(',') + ']';
};
class ServerFetchAdapter {
    #config;
    #logger;
    constructor(config, logger) {
        this.#config = config;
        this.#logger = logger;
    }
    async _invoke(url, options) {
        const headers = {
            ...options.headers,
            ...this.#config.headers,
        };
        if (options.contentType) {
            headers['Content-Type'] = options.contentType;
        }
        else if (typeof options.body === 'string' ||
            options.body instanceof Uint8Array ||
            options.body instanceof ArrayBuffer) {
            headers['Content-Type'] = 'application/octet-stream';
        }
        const method = options.method ?? 'POST';
        this.#logger.trace('sending %s to %s with headers: %s', method, url, redactHeaders(headers));
        const res = await fetch(url, {
            method,
            body: options.body,
            headers,
            signal: options.signal,
            ...(options.duplex ? { duplex: options.duplex } : {}),
        });
        if (res.ok) {
            switch (res.status) {
                case 100:
                case 101:
                case 102:
                case 204:
                case 304:
                    return {
                        ok: true,
                        data: undefined,
                        response: res,
                    };
                default:
                    break;
            }
            if (options?.binary) {
                return {
                    ok: true,
                    data: undefined,
                    response: res,
                };
            }
            const data = await fromResponse(res);
            return {
                ok: true,
                data,
                response: res,
            };
        }
        if (res.status === 404) {
            return {
                ok: false,
                response: res,
            };
        }
        const err = await toServiceException(method, url, res);
        throw err;
    }
    async invoke(url, options = { method: 'POST' }) {
        if (this.#config.onBefore) {
            let result = undefined;
            let err = undefined;
            await this.#config.onBefore(url, options, async () => {
                try {
                    result = await this._invoke(url, options);
                    if (this.#config.onAfter) {
                        await this.#config.onAfter(url, options, result);
                    }
                }
                catch (ex) {
                    err = ex;
                    if (this.#config.onAfter && err instanceof ServiceException) {
                        await this.#config.onAfter(url, options, {
                            ok: false,
                            response: new Response(err.message, {
                                status: err.statusCode,
                            }),
                        }, err);
                    }
                }
            });
            if (err) {
                throw err;
            }
            return result;
        }
        else {
            return await this._invoke(url, options);
        }
    }
}
/**
 * Create a Server Side Fetch Adapter to allow the server to add headers and track outgoing requests
 *
 * @param config the service config
 * @returns
 */
export function createServerFetchAdapter(config, logger) {
    return new ServerFetchAdapter(config, logger);
}
//# sourceMappingURL=server.js.map