import { type InferOutput, type StandardSchemaV1 } from '@agentuity/core';
import type { Handler } from 'hono/types';
/**
 * Route validator for standard HTTP routes (non-agent routes).
 * Provides input and output validation with full type safety.
 *
 * **Input validation behavior by HTTP method:**
 * - **GET**: Validates query parameters (accessible via `c.req.valid('query')`)
 * - **POST/PUT/PATCH/DELETE**: Validates JSON body (accessible via `c.req.valid('json')`)
 *
 * **Output validation:**
 * - Validates response JSON body before sending
 * - Throws 500 Internal Server Error if validation fails
 *
 * @template TInput - Input schema type (query params for GET, body for POST/PUT/PATCH/DELETE)
 * @template TOutput - Output schema type for response validation
 *
 * @example GET route with output validation only
 * ```typescript
 * router.get('/users',
 *   validator({ output: z.array(UserSchema) }),
 *   async (c) => {
 *     return c.json([{ id: '1', name: 'Alice' }]);
 *   }
 * );
 * ```
 *
 * @example GET route with query param validation
 * ```typescript
 * router.get('/users/:id',
 *   validator({
 *     input: z.object({ id: z.string() }),
 *     output: UserSchema
 *   }),
 *   async (c) => {
 *     const { id } = c.req.valid('query'); // Fully typed!
 *     return c.json({ id, name: 'Alice' });
 *   }
 * );
 * ```
 *
 * @example POST route with body and output validation
 * ```typescript
 * router.post('/users',
 *   validator({
 *     input: z.object({ name: z.string(), email: z.string().email() }),
 *     output: UserSchema
 *   }),
 *   async (c) => {
 *     const data = c.req.valid('json'); // Fully typed!
 *     return c.json({ id: '1', ...data });
 *   }
 * );
 * ```
 */
export interface RouteValidator {
    /**
     * Output-only validation (no input validation).
     * Useful for GET routes without query parameters.
     *
     * @template TOutput - Output schema type
     * @param options - Configuration object with output schema
     * @returns Hono middleware handler
     */
    <TOutput extends StandardSchemaV1>(options: {
        output: TOutput;
        stream?: boolean;
    }): Handler<any, any, {
        in: {};
        out: {
            json: InferOutput<TOutput>;
        };
    }>;
    /**
     * Input and output validation.
     * - GET: validates query parameters
     * - POST/PUT/PATCH/DELETE: validates JSON body
     *
     * @template TInput - Input schema type
     * @template TOutput - Optional output schema type
     * @param options - Configuration object with input and optional output schemas
     * @returns Hono middleware handler with type inference
     */
    <TInput extends StandardSchemaV1, TOutput extends StandardSchemaV1 | undefined = undefined>(options: {
        input: TInput;
        output?: TOutput;
        stream?: boolean;
    }): Handler<any, any, {
        in: {};
        out: {
            json: InferOutput<TInput>;
            query: InferOutput<TInput>;
        };
    }>;
}
/**
 * Create a route validator middleware with input and/or output validation.
 *
 * Automatically handles different validation targets based on HTTP method:
 * - GET requests: validates query parameters (if input schema provided)
 * - POST/PUT/PATCH/DELETE: validates JSON body (if input schema provided)
 * - All methods: validates JSON response (if output schema provided)
 *
 * @param options - Validation configuration
 * @param options.input - Input schema (query params for GET, body for POST/PUT/PATCH/DELETE)
 * @param options.output - Output schema for response validation
 * @returns Hono middleware handler
 *
 * @example GET with query validation
 * ```typescript
 * import { validator } from '@agentuity/runtime';
 *
 * router.get('/search',
 *   validator({
 *     input: z.object({ q: z.string(), limit: z.number().optional() }),
 *     output: z.array(SearchResultSchema)
 *   }),
 *   async (c) => {
 *     const { q, limit } = c.req.valid('query'); // Typed!
 *     const results = await search(q, limit);
 *     return c.json(results);
 *   }
 * );
 * ```
 *
 * @example POST with body validation
 * ```typescript
 * router.post('/users',
 *   validator({
 *     input: z.object({ name: z.string() }),
 *     output: UserSchema
 *   }),
 *   async (c) => {
 *     const data = c.req.valid('json'); // Typed!
 *     const user = await createUser(data);
 *     return c.json(user);
 *   }
 * );
 * ```
 */
export declare const validator: RouteValidator;
//# sourceMappingURL=validator.d.ts.map